﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;

namespace csharp_CS_API_Test
{
    class Functions
    {
        [DllImport("msvcrt")]
        static extern int _getch();
        static public CChromasensCameraIf m_camera;
        static public bool m_isConnected;

        public static void printAPIError (short nErrorNo)
        {
            API_Test.textattr(API_Test.Color.RED);
            Console.WriteLine("Sending of HSI-File failed!\n");
            StringBuilder errorText = new StringBuilder("", CS_API.MAX_ERROR_TEXT_LENGTH);
            StringBuilder camState = new StringBuilder("", CS_API.MAX_ERROR_TEXT_LENGTH);
            short nErrorCode = m_camera.get_camera_state(true, camState);
            short res = m_camera.get_error_text(nErrorCode, errorText);
            Console.WriteLine("Error({0}):{1}",nErrorCode, errorText);
        }

        //////////////////////////////////////////////////////////////////////////////////////
        /// Function will return the currently active setting number
        //////////////////////////////////////////////////////////////////////////////////////
        public static Int16 getCurrentSettingNumber()
        {
            if (m_isConnected)
            {
                return m_camera.get_current_setting_number();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
            
            return CS_API.CS_GENERAL_ERROR;
        }

        //////////////////////////////////////////////////////////////////////////////////////
        /// To load the Chromasens API, call this function
        //////////////////////////////////////////////////////////////////////////////////////
        static public Int16 InitAPI()
        {
            Int16 nReturn = CS_API.CS_GENERAL_ERROR;
            // Use the desired configuration directory as parameter if another directory as ./conf is used to store the configuration files
            m_camera = CS_API.InitCsCameraIf("");
            if (m_camera != null)
            {
                nReturn = CS_API.CS_OK;
            }
            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////////////////
        /// Return the connection state
        //////////////////////////////////////////////////////////////////////////////////////
        static public bool isConnected()
        {
            return m_isConnected;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Close the camera and freee the ressources
        //////////////////////////////////////////////////////////////////////////
        static public Int16 closeInterface()
        {
            Int16 nReturn = CS_API.CS_OK;
            // Close the interface again
            if (m_camera != null)
            {
                nReturn = m_camera.close_control();
                if (m_camera != null)
                {
                    m_camera.Dispose();
                    m_camera = null;
                }
            }
            return nReturn;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Resets the camera and establishes a new connection to it
        //////////////////////////////////////////////////////////////////////////
        static public Int16 resetCamera()
        {
            Int16 nReturn = CS_API.CS_GENERAL_ERROR;
            if (m_isConnected)
            {
                nReturn = m_camera.reset_camera();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
            return nReturn;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Handles the connection to the m_camera
        //////////////////////////////////////////////////////////////////////////
        static public int connectCamera()
        {
            CS_CONNECTION_STRUCT connectInfo = new CS_CONNECTION_STRUCT();
            short nReturn = CS_API.CS_GENERAL_ERROR;
            list_availableInterfaces();
            Console.WriteLine("\nChoose the type of connection:");
            Console.WriteLine("(0) Choose yourself:");
            Console.WriteLine("(1) RS232");
            Console.WriteLine("(2) CameraLink");
            Console.WriteLine("(q) Quit");
            int nChoice = _getch();
            switch (nChoice)
            {
                case '0':
                    nReturn = m_camera.open_control(connectInfo);
                    break;
                case '1':
                    // Directly open a connection via cameraLink
                    {
                        short nPort = 0;
                        int nBaudRate = 115200;
                        Console.Write("\n\nPlease enter the desired COM-port: ");
                        nPort = Convert.ToInt16(Console.ReadLine());
                        Console.Write("Please enter the desired BAUD-Rate: ");
                        nBaudRate = Convert.ToInt32(Console.ReadLine());
                        nReturn = m_camera.open_RS232_control(nPort, nBaudRate);
                    }
                    break;
                case '2':
                    {
                        // Directly connect to the m_camera via a RS232 connection(Virtual com ports are also permissible
                        short nPort = 0;
                        int nBaudRate = 115200;
                        Console.Write("\nPlease enter the desired CameraLink-port: ");
                        nPort = Convert.ToInt16(Console.ReadLine());
                        Console.Write("Please enter the desired BAUD-Rate: ");
                        nBaudRate = Convert.ToInt32(Console.ReadLine());
                        nReturn = m_camera.open_CL_control(nPort, nBaudRate);
                    }
                    break;
                default:
                    m_isConnected = false;
                    break;
            }
            // check if the connection was successful
            if (nReturn == CS_API.CS_OK)
            {
                Console.Write("\nSuccessfully connected with: " + connectInfo.connectionDescription);
                m_isConnected = true;
            }
            else
            {
                Console.Write("\nFailed to connect to m_camera error code: " + nReturn);
                m_isConnected = false;
            }
            connectInfo.Dispose();
            return nReturn;
        }


        //////////////////////////////////////////////////////////////////////////
        /// Set the initial gain values of the allPIXA m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 setInitialGainValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_CHANNEL_STRUCT initialGain = new CS_CHANNEL_STRUCT();
                if (m_camera.get_initial_gain_values(initialGain) == CS_API.CS_OK)
                {
                    Console.Write("\n");
                    for (int i = 0; i < initialGain.no_of_valid_entries; i++)
                    {
                        UInt16 nTmp = 0;
                        Console.Write("Please enter the intial gain value No." + i + " (decimal): ");
                        nTmp = Convert.ToUInt16(Console.ReadLine());
                        if (nTmp <= 0)
                            nTmp = 0;
                        initialGain.set_value(i, nTmp);
                    }
                    m_camera.set_initial_gain_values(initialGain);
                }
                initialGain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;

        }


        //////////////////////////////////////////////////////////////////////////
        /// Gets the initial gain values of the allPIXA m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 getInitialGainValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_CHANNEL_STRUCT initialGain = new CS_CHANNEL_STRUCT();
                if (m_camera.get_initial_gain_values(initialGain, true) == CS_API.CS_OK)
                {
                    Console.Write("\n\nThe values of the initial gain settings:\n");
                    for (int i = 0; i < initialGain.no_of_valid_entries; i++)
                    {
                        Console.WriteLine("Element " + i + "  " + " : " + initialGain.get_value(i));
                    }
                }
                initialGain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the gain values of the allPIXA m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 setGainValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_CHANNEL_STRUCT gain = new CS_CHANNEL_STRUCT();

                if (m_camera.get_gain_values(gain) == CS_API.CS_OK)
                {
                    Console.Write("\n");
                    for (int i = 0; i < gain.no_of_valid_entries; i++)
                    {
                        UInt16 nTmp = 0;
                        Console.Write("Please enter the Gain-Value No." + i + " (decimal): ");
                        nTmp = Convert.ToUInt16(Console.ReadLine());
                        if (nTmp <= 0)
                            nTmp = 0;

                        gain.set_value(i, nTmp);
                    }
                    m_camera.set_gain_values(gain);
                }
                gain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the gain values of the allPIXA m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 getGainValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_CHANNEL_STRUCT gain = new CS_CHANNEL_STRUCT();
                if (m_camera.get_gain_values(gain, true) == CS_API.CS_OK)
                {
                    Console.Write("\n\nThe values of the gain settings:\n");
                    for (int i = 0; i < gain.no_of_valid_entries; i++)
                    {
                        Console.WriteLine("Element " + i + "  " + " : " + gain.get_value(i));
                    }
                }
                gain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the linear gain values of the allPIXA wave m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 setLinearGainValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_LINEAR_GAIN_STRUCT linearGain = new CS_LINEAR_GAIN_STRUCT();

                if (m_camera.get_linear_gain_values(linearGain) == CS_API.CS_OK)
                {
                    Console.Write("\n");
                    for (int i = 0; i < linearGain.no_of_valid_entries; i++)
                    {
                        Double nTmp = 0;
                        Console.Write("Please enter the Gain-Value No." + i + " (double): ");
                        nTmp = Convert.ToDouble(Console.ReadLine());
                        if (nTmp <= 0)
                            nTmp = 0;

                        linearGain.set_value(i, nTmp);
                    }
                    m_camera.set_linear_gain_values(linearGain);
                }
                linearGain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the linear gain values of the allPIXA wave m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 getLinearGainValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_LINEAR_GAIN_STRUCT linearGain = new CS_LINEAR_GAIN_STRUCT();
                if (m_camera.get_linear_gain_values(linearGain, true) == CS_API.CS_OK)
                {
                    Console.Write("\n\nThe values of the gain settings:\n");
                    for (int i = 0; i < linearGain.no_of_valid_entries; i++)
                    {
                        Console.WriteLine("Element " + i + "  " + " : " + linearGain.get_value(i));
                    }
                }
                linearGain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the analog coarse gain values of the allPIXA m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 getCDSGainValues()
        {
            if(m_isConnected)
            {
                // get the available number of gain values
                CS_ANALOG_COARSE_GAIN_STRUCT cdsGain = new CS_ANALOG_COARSE_GAIN_STRUCT();
                if(m_camera.get_coarse_gain_values(cdsGain, true) == CS_API.CS_OK)
                {
                    //char []txt = new char[200];
                    string txt = "";
                    Console.Write("\n\nThe values of the coarse gainsettings:\n");
                    for(int i = 0; i < cdsGain.no_of_valid_entries; i++)
                    {
                        switch(cdsGain.get_value(i)){
                            case (int) CS_COARSE_GAIN.MINUS_3DB:
                                txt = "-3 dB";
                                break;
                            case (int) CS_COARSE_GAIN._0_DB:
                                txt = " 0 dB";
                                break;
                            case (int) CS_COARSE_GAIN._3_DB:
                                txt = " 3 dB";
                                break;
                            case (int) CS_COARSE_GAIN._6_DB:
                                txt = " 6 dB";
                                break;
                            default:
                                Console.Write("invalid entry! ");
                                break;
                        }
                        Console.Write("Element " + i + ": " + txt + "\n");
                    }
                }
                cdsGain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }
        //////////////////////////////////////////////////////////////////////////
        /// Set the analog coarse gain values of the allPIXA m_camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 setCDSGainValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_ANALOG_COARSE_GAIN_STRUCT cdsGain = new CS_ANALOG_COARSE_GAIN_STRUCT();
                if (m_camera.get_coarse_gain_values(cdsGain) == CS_API.CS_OK)
                {
                    Console.Write("Please enter one of the following values(-3, 0, 3 or 6)[dB]\n");
                    for (int i = 0; i < cdsGain.no_of_valid_entries; i++)
                    {
                        Int16 nTmp = 0;
                        Console.Write("Please enter the coarse gain-Value No." + i + " (decimal): ");
                        nTmp = Convert.ToInt16(Console.ReadLine());
                        //if (nTmp <= 0)
                        //    nTmp = 0;
                        switch (nTmp)
                        {
                            case -3:
                                nTmp = (int) CS_COARSE_GAIN.MINUS_3DB;
                                break;
                            case 0:
                                nTmp = (int) CS_COARSE_GAIN._0_DB;
                                break;
                            case 3:
                                nTmp = (int) CS_COARSE_GAIN._3_DB;
                                break;
                            case 6:
                                nTmp = (int) CS_COARSE_GAIN._6_DB;
                                break;
                            default:
                                Console.Write("invalid entry! Will set the value to 0 dB");
                                nTmp = (int) CS_COARSE_GAIN._0_DB;
                                break;
                        }
                        cdsGain.set_value(i, nTmp);
                    }
                    m_camera.set_coarse_gain_values(cdsGain);
                }
                cdsGain.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }


        //////////////////////////////////////////////////////////////////////////
        /// Gets the operational values of the camera
        //////////////////////////////////////////////////////////////////////////
        static public Int16 get_camera_operating_values()
        {
            if (m_isConnected)
            {
                CS_CAMERA_OPERATING_VALUES_STRUCT operatingValues = new CS_CAMERA_OPERATING_VALUES_STRUCT();
                if (m_camera.get_camera_operating_values(operatingValues) == CS_API.CS_OK)
                {
                    Console.Write("\n\nInput state: 0x" + operatingValues.inputStatus);
                    Console.Write("\ngain control disable state: 0x" + operatingValues.gainCtrlDisableState);
                    Console.Write("\nimage count: " + operatingValues.imageCount);
                    Console.Write("\ncurrent wref values:");
                    Console.Write("\nRO:        " + operatingValues.get_nCurrentWhiteRef(CS_API.RED_ODD) + "   RE:         " + operatingValues.get_nCurrentWhiteRef(CS_API.RED_EVEN));
                    Console.Write("\nGO_REAR: " + operatingValues.get_nCurrentWhiteRef(CS_API.RED_ODD_REAR) + "   RE_REAR:    " + operatingValues.get_nCurrentWhiteRef(CS_API.RED_EVEN_REAR));
                    Console.Write("\nGO:      " + operatingValues.get_nCurrentWhiteRef(CS_API.GREEN_ODD) + "   GE:         " + operatingValues.get_nCurrentWhiteRef(CS_API.GREEN_EVEN));
                    Console.Write("\nGO_REAR: " + operatingValues.get_nCurrentWhiteRef(CS_API.GREEN_ODD_REAR) + "   GE_REAR:    " + operatingValues.get_nCurrentWhiteRef(CS_API.GREEN_EVEN_REAR));
                    Console.Write("\nBO:      " + operatingValues.get_nCurrentWhiteRef(CS_API.BLUE_ODD) + "  BE:         " + operatingValues.get_nCurrentWhiteRef(CS_API.BLUE_EVEN));
                    Console.Write("\nBO_REAR: " + operatingValues.get_nCurrentWhiteRef(CS_API.BLUE_ODD_REAR) + "   BE_REAR:    " + operatingValues.get_nCurrentWhiteRef(CS_API.BLUE_EVEN_REAR));
                    Console.Write("\ncurrent gain values:");
                    Console.Write("\nRO:      " + operatingValues.get_nCurrentGain(CS_API.RED_ODD) + "   RE:         " + operatingValues.get_nCurrentGain(CS_API.RED_EVEN));
                    Console.Write("\nGO_REAR: " + operatingValues.get_nCurrentGain(CS_API.RED_ODD_REAR) + "   RE_REAR:    " + operatingValues.get_nCurrentGain(CS_API.RED_EVEN_REAR));
                    Console.Write("\nGO:      " + operatingValues.get_nCurrentGain(CS_API.GREEN_ODD) + "   GE:         " + operatingValues.get_nCurrentGain(CS_API.GREEN_EVEN));
                    Console.Write("\nGO_REAR: " + operatingValues.get_nCurrentGain(CS_API.GREEN_ODD_REAR) + "   GE_REAR:    " + operatingValues.get_nCurrentGain(CS_API.GREEN_EVEN_REAR));
                    Console.Write("\nBO:      " + operatingValues.get_nCurrentGain(CS_API.BLUE_ODD) + "   BE:         " + operatingValues.get_nCurrentGain(CS_API.BLUE_EVEN));
                    Console.Write("\nBO_REAR: " + operatingValues.get_nCurrentGain(CS_API.BLUE_ODD_REAR) + "   BE_REAR:    " + operatingValues.get_nCurrentGain(CS_API.BLUE_EVEN_REAR));
                    Console.Write("\nsync integration time: " + operatingValues.syncIntegrationTime);
                    Console.Write("\ncamera state: 0x" + operatingValues.cameraState);
                    Console.Write("\nerror state: 0x" + operatingValues.errorState);
                    Console.Write("\nled state: 0x" + operatingValues.ledState);
                    if (operatingValues.currentTransportSpeed == CS_API.SPEED_2_HIGH)
                        Console.Write("\ntransport speed too high: " + operatingValues.currentTransportSpeed);
                    else if (operatingValues.currentTransportSpeed == CS_API.SPEED_2_SLOW)
                        Console.Write("\ntransport speed too slow: " + operatingValues.currentTransportSpeed);
                    else 
                        Console.Write("\ntransport speed: " + operatingValues.currentTransportSpeed);
                    Console.Write("\nanalog1:   " + operatingValues.get_internalVoltage(0));
                    Console.Write("\nanalog2:   " + operatingValues.get_internalVoltage(1));
                    Console.Write("\nVcore:     " + operatingValues.get_internalVoltage(2));
                    Console.Write("\nsupply1:   " + operatingValues.get_internalVoltage(3));
                    Console.Write("\nsupply2:   " + operatingValues.get_internalVoltage(4));
                    Console.Write("\nsupplyCCD: " + operatingValues.get_internalVoltage(5));
                    Console.Write("\nLedVoltage_0: " + operatingValues.get_voltageLed(0));
                    Console.Write("\nLedVoltage_1: " + operatingValues.get_voltageLed(1));
                    Console.Write("\nLedVoltage_2: " + operatingValues.get_voltageLed(2));
                    Console.Write("\nLedVoltage_3: " + operatingValues.get_voltageLed(3));
                    Console.Write("\nLedVoltage_4: " + operatingValues.get_voltageLed(4));
                    Console.Write("\nInput Voltage: " + operatingValues.inputVoltage);
                    Console.Write("\ntemperature board: " + operatingValues.temperatureBoard);
                    Console.Write("\ntemperature led controller: " + operatingValues.temperatureLEDController);
                    Console.Write("\ntemperature sensor: " + operatingValues.temperatureSensor);
                    Console.Write("\ntemperature led: " + operatingValues.temperatureLed);
                }
                else
                {
                    Console.Write("\n\nError getting operation values from camera");
                }
                operatingValues.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the currently active settings for brightness, contrast and gamma correction
        //////////////////////////////////////////////////////////////////////////
        static public Int16 setBrightnessAndContrast()

        {
            Int16 tmp=0; // Buffer variable because scanf is using 32 Bit values!
            double dTmp;
            if(m_isConnected)
            {
                CS_BRIGHTNESS_CONTRAST_STRUCT brightnessContrast = new CS_BRIGHTNESS_CONTRAST_STRUCT();
 
                Console.Write("\n\nUse brightness and contrast(0=no or 1=yes): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if(tmp <= 0)
                    tmp = 0;
                
                brightnessContrast.useBrightnessAndContrast = (0 !=  tmp);

                Console.Write("Brightness (red):\n");
                tmp = Convert.ToInt16(Console.ReadLine());
                brightnessContrast.set_brightness(0, tmp);
                Console.Write("Brightness (green):\n");
                tmp = Convert.ToInt16(Console.ReadLine());

                brightnessContrast.set_brightness(1, tmp);
                Console.Write("Brightness (blue):\n");
                tmp = Convert.ToInt16(Console.ReadLine());
                brightnessContrast.set_brightness(2, tmp);
                Console.Write("Contrast (red):\n");

                dTmp = Convert.ToDouble(Console.ReadLine());
                brightnessContrast.set_contrast(0, dTmp + 0.0005);
                Console.Write("Contrast (green):\n");
                dTmp = Convert.ToDouble(Console.ReadLine());
               brightnessContrast.set_contrast(1, dTmp + 0.0005);
                
                Console.Write("Contrast (blue):\n");
                dTmp = Convert.ToDouble(Console.ReadLine());
                
                brightnessContrast.set_contrast(2, dTmp + 0.0005);

                Console.Write("Gamma:\n");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0)
                    dTmp = 0.0;
                brightnessContrast.gamma_correction = (double) (dTmp + 0.05);
                
                Int16 nError = m_camera.set_brightness_and_contrast(brightnessContrast);
                if( nError != CS_API.CS_OK)
                {
                    Console.Write("Error setting the brightness and contrast settings in the camera \n" + nError);
                }
                brightnessContrast.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        
        }


        //////////////////////////////////////////////////////////////////////////
        /// Sets brightness, contrast and gamma correction
        //////////////////////////////////////////////////////////////////////////
        static public Int16 getBrightnessAndContrast()
        {
            if(m_isConnected)
            {
                CS_BRIGHTNESS_CONTRAST_STRUCT brightnessContrast = new CS_BRIGHTNESS_CONTRAST_STRUCT();
                if(m_camera.get_brightness_and_contrast(brightnessContrast, true) == CS_API.CS_OK)
                {
                    Console.Write("\n\nUse brightness and contrast: " + brightnessContrast.useBrightnessAndContrast);
                    Console.Write("\nBrightness(red): " + brightnessContrast.get_brightness(0));
                    Console.Write("\nBrightness(green): " + brightnessContrast.get_brightness(1));
                    Console.Write("\nBrightness(blue): " + brightnessContrast.get_brightness(2));
                    Console.Write("\nContrast(red): " + brightnessContrast.get_contrast(0));
                    Console.Write("\nContrast(green): " + brightnessContrast.get_contrast(1));
                    Console.Write("\nContrast(blue): " + brightnessContrast.get_contrast(2));
                    Console.Write("\nGamma: " + brightnessContrast.gamma_correction);
                }
                else
                {
                    Console.Write("\n\nError getting brightness and contrast settings from camera");
                }
                brightnessContrast.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;

        }

        //////////////////////////////////////////////////////////////////////////
        /// Returns the information about serial number, firmware and fpga-version
        //////////////////////////////////////////////////////////////////////////
        static public Int16 get_camera_information()
        {
            if (m_isConnected)
            {
                CS_CAMERA_INFORMATION_STRUCT cameraInfo = new CS_CAMERA_INFORMATION_STRUCT();
                if (m_camera.get_camera_information(cameraInfo) == CS_API.CS_OK)
                {
                    Console.Write("\n\nFirmware version: " + cameraInfo.get_firmwareVersion(0) + "-" + cameraInfo.get_firmwareVersion(1) + "-" + cameraInfo.get_firmwareVersion(2));
                    Console.Write("\nFPGA version: " + cameraInfo.fpgaVersion);
                    Console.Write("\nFirmware description: " + cameraInfo.firmwareDescription);
                    Console.Write("\nFPGA description: " + cameraInfo.fpgaDescription);
                    Console.Write("\nCamera serial number: " + cameraInfo.cameraSerialNumber);
                }
                else
                {
                    Console.Write("\n\nError getting operation values from camera");
                }
                cameraInfo.Dispose();
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

            return CS_API.CS_OK;
        }

        //////////////////////////////////////////////////////////////////////////
        // Read a bin value from the m_camera
        // The get function by default does not request another response from the m_camera
        // The value read will be the value obtained by the last get call with the update parameter set to true
        //////////////////////////////////////////////////////////////////////////
        public static void getBinTag()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the tag ID (BIN) to get(in HEX): ");
                string hexValue = Console.ReadLine();
                ushort tagId = (ushort) int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tagId != 0)
                {
                    bool binVal=false;
                    if (m_camera.get_bin_tag(tagId, ref binVal, true) == CS_API.CS_OK)
                    {
                        Console.Write("\nRead BIN Tag (" + hexValue + "): " + binVal);
                    }
                    else
                    {
                        Console.Write("\n\nError getting BIN-Tag from camera");
                    }
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Prompts the user to input the data for the sending of an bin tag
        //////////////////////////////////////////////////////////////////////////
        public static void setBinTag()
        {
            Console.Write("Please enter the tag ID (BIN) to set(in HEX): ");
            string hexValue = Console.ReadLine();
            ushort tagId = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
            if (tagId != 0)
            {
                bool binVal = false;
                Console.Write("Please enter the value to set(0 or 1): ");
                binVal = (Convert.ToUInt16(Console.ReadLine()) != 0);
                if (m_camera.send_bin_tag(tagId, binVal, true) == CS_API.CS_OK)
                {
                    Console.WriteLine("Send bin Tag successfully!");
                }
                else
                {
                    Console.Write("\n\nError sending BIN-Tag to camera");
                }
            }
        }

        //////////////////////////////////////////////////////////////////////////
        /// Reads a given short tag from the m_camera
        //////////////////////////////////////////////////////////////////////////
        public static void getShortTag()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the tag ID (SHORT) to get(in HEX): ");
                string hexValue = Console.ReadLine();
                ushort tagId = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tagId != 0)
                {
                    ushort shortVal = 0;
                    if (m_camera.get_short_tag(tagId, ref shortVal, true) == CS_API.CS_OK)
                    {
                        Console.Write("\nRead LONG Tag (" + hexValue + "): " + shortVal);
                    }
                    else
                    {
                        Console.Write("\n\nError getting SHORT-Tag from camera");
                    }
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Prompts the user to input the data for the sending of an short tag
        //////////////////////////////////////////////////////////////////////////
        public static void setShortTag()
        {
            Console.Write("Please enter the tag ID (SHORT) to set(in HEX): ");
            string hexValue = Console.ReadLine();
            ushort tagId = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
            if (tagId != 0)
            {
                UInt16 shortVal = 0;
                Console.Write("Please enter the value to set(decimal): ");
                shortVal = Convert.ToUInt16(Console.ReadLine());
                if (m_camera.send_short_tag(tagId, shortVal, true) == CS_API.CS_OK)
                {
                    Console.WriteLine("Send short Tag successfully!");
                }
                else
                {
                    Console.Write("\n\nError sending SHORT-Tag to camera");
                }
            }
        }

        //////////////////////////////////////////////////////////////////////////
        /// Reads a given long tag from the m_camera
        //////////////////////////////////////////////////////////////////////////
        public static void getLongTag()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the tag ID (LONG) to get(in HEX): ");
                string hexValue = Console.ReadLine();
                ushort tagId = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tagId != 0)
                {
                    UInt32 longVal = 0;
                    if (m_camera.get_long_tag(tagId, ref longVal, true) == CS_API.CS_OK)
                    {
                        Console.Write("\nRead LONG Tag (" + hexValue + "): " + longVal);
                    }
                    else
                    {
                        Console.Write("\n\nError getting LONG-Tag from camera");
                    }
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Prompts the user to input the data for the sending of an long tag
        //////////////////////////////////////////////////////////////////////////
        public static void setLongTag()
        {
            Console.Write("Please enter the tag ID (LONG) to set(in HEX): ");
            string hexValue = Console.ReadLine();
            ushort tagId = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
            if (tagId != 0)
            {
                UInt32 longVal = 0;
                Console.Write("Please enter the value to set(decimal): ");
                longVal = Convert.ToUInt32(Console.ReadLine());
                if (m_camera.send_long_tag(tagId, longVal, true) == CS_API.CS_OK)
                {
                    Console.WriteLine("Send long Tag successfully!");
                }
                else
                {
                    Console.Write("\n\nError sending LONG-Tag to camera");
                }
            }
        }

        //////////////////////////////////////////////////////////////////////////
        /// Reads a given var tag from the m_camera
        //////////////////////////////////////////////////////////////////////////
        public static void getVarTag()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the tag ID (VAR) to get(in HEX): ");
                string hexValue = Console.ReadLine();
                ushort tagId = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tagId != 0)
                {
                    Console.Write("Please enter the length of the var tag: ");
                    UInt16 varLength = Convert.ToUInt16( Console.ReadLine());
                    if (varLength > 0)
                    {
                        UInt16[] varValues = new UInt16[varLength];
                        //SWIGTYPE_p_unsigned_short varPointer = ref ptr;//(ref varValues);

                        if (m_camera.get_var_tag(tagId, varLength, varValues, true) == CS_API.CS_OK)
                        {
                            Console.Write("\nRead VAR Tag (" + hexValue + "): ");
                            int count = 0;
                            foreach (UInt16 element in varValues)
                            {
                                count += 1;
                                System.Console.WriteLine("Element #{0}: {1}", count, element);
                            }
                        }
                        else
                        {
                            Console.Write("\n\nError getting VAR-Tag from camera");
                        }
                    }
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Prompts the user to input the data for the sending of an var tag
        //////////////////////////////////////////////////////////////////////////
        public static void setVarTag()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the tag ID (LONG) to set(in HEX): ");
                string hexValue = Console.ReadLine();
                ushort tagId = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tagId != 0)
                {
                    Console.Write("Please enter the length of the var tag: ");
                    UInt16 varLength = Convert.ToUInt16(Console.ReadLine());
                    if (varLength > 0)
                    {
                        UInt16[] varValues = new UInt16[varLength];
                        int count = 0;
                        foreach (UInt16 element in varValues)
                        {
                            System.Console.Write("Element #{0}: ", count);
                            varValues[count] = Convert.ToUInt16(Console.ReadLine());
                            count += 1;
                        }
                        if (m_camera.send_var_tag(tagId, varLength, varValues, true) == CS_API.CS_OK)
                        {
                            Console.WriteLine("Send VAR Tag successfully!");
                        }
                        else
                        {
                            Console.Write("\n\nError sending VAR-Tag to camera");
                        }
                    }
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sends a given HSI file to the camera
        //////////////////////////////////////////////////////////////////////////
        public unsafe static void sendHSIFile()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the file name to send(including path):\n ");
                string fileName = Console.ReadLine();
                if (fileName.Length > 0)
                {
                    short nErrorNo = m_camera.send_hsi_file_2_camera(fileName);
                    if (nErrorNo == CS_API.CS_OK)
                    {
                        Console.WriteLine("Successfully sent file to camera");
                    }
                    else
                    {
                        API_Test.textattr(API_Test.Color.RED);
                        Console.WriteLine("Sending of HSI-File failed!\n");
                        StringBuilder errorText = new StringBuilder("", CS_API.MAX_ERROR_TEXT_LENGTH);
                        short res = m_camera.get_error_text(nErrorNo, errorText);
                        Console.WriteLine("Error:" + errorText);
                    }
                }
                else
                {
                    API_Test.textattr(API_Test.Color.RED);
                    Console.Write("\n\nNo valid filename was given!");
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Function will select a different setting
        //////////////////////////////////////////////////////////////////////////
        public static void selectSetting()
        {
            if (m_isConnected)
            {
                Console.WriteLine("Please enter the setting to select: ");
                short settingNo = Convert.ToInt16(Console.ReadLine());
                if ((settingNo >= CS_API.MIN_SETTING_NO) && (settingNo <= CS_API.MAX_SETTING_NO))
                {
                    Int32 nReturn = m_camera.select_active_setting(settingNo);
                    if (nReturn != CS_API.CS_OK)
                    {
                        API_Test.textattr(API_Test.Color.RED);
                        if (nReturn == CS_API.CS_ERROR_SETTING_NO_OUT_OF_RANGE)
                            Console.WriteLine("\nError setting active setting: Setting number out of range\n");
                        else if (nReturn == CS_API.CS_ERROR_HSI_SEND)
                            Console.WriteLine("\nError setting active setting: Send of HSI-command failed\n");
                        else if (nReturn == CS_API.CS_GENERAL_ERROR)
                            Console.WriteLine("\nError setting active setting: Unspecified error\n");
                        else if (nReturn == CS_API.CS_ERROR_SETTING_NOT_AVAILABLE)
                            Console.WriteLine("\nError setting does not exist on the m_camera\n");
                        else
                            Console.WriteLine("\nError setting active setting: Unknown error\n");
                    }
                    else
                        Console.WriteLine("Successfully activated setting No: {0}", settingNo);
                }
                else
                {
                    API_Test.textattr(API_Test.Color.RED);
                    Console.WriteLine("\nError! Please enter a number in between {0} and {1}!\n", CS_API.MIN_SETTING_NO, CS_API.MAX_SETTING_NO);
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Function will print the available settings of the m_camera
        //////////////////////////////////////////////////////////////////////////
        public static void showAvailableSettings()
        {
            if (m_isConnected)
            {
                long nReturn = m_camera.get_available_settings();
                if (nReturn < CS_API.CS_OK)
                {
                    API_Test.textattr(API_Test.Color.RED);
                    Console.WriteLine("\nError getting available settings: {0}\n", nReturn);
                }
                else
                {
                    Console.WriteLine("Available Settings on the camera:");
                    Int64 one = 1;
                    for (ushort i = CS_API.MIN_SETTING_NO; i <= CS_API.MAX_SETTING_NO; i++)
                    {
                        Int64 tmp = (one << i);
                        if ((nReturn & tmp)> 0)
                        {
                            Console.Write("{0}, ", i);
                        }
                    }
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Function will burn the current setting to the m_camera (if desired to another setting)
        //////////////////////////////////////////////////////////////////////////
        public static void burnSettingsToCamera()
        {
            if (m_isConnected)
            {
                int nCurrentSetting = m_camera.get_current_setting_number();
                Console.WriteLine("\nDo you want to permanently burn the current setting ({0}) to the camera? (y/n)", nCurrentSetting);
                if (Console.ReadLine() == "y")
                {
                    Console.WriteLine("\nPlease enter the setting number to burn the setting to (current setting: {0})", nCurrentSetting);
                    short settingNo = Convert.ToInt16(Console.ReadLine());
                    if((settingNo >= CS_API.MIN_SETTING_NO) && (settingNo <= CS_API.MAX_SETTING_NO))
                    {
                        short nReturn = m_camera.burn_current_setting_to_camera(settingNo);
                        if (nReturn != CS_API.CS_OK)
                        {
                            printAPIError(nReturn);
                        }
    
                    }
                    else
                    {
                        Console.WriteLine("Setting number out of range!");
                    }
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }


        //////////////////////////////////////////////////////////////////////////
        /// Function will save the current setting to disk
        /// the output format can be chosen in between binary HSI-format or XML-format used in the CST
        //////////////////////////////////////////////////////////////////////////
        public static void saveCurrentSetting()
        {
            if (m_isConnected)
            {
                int nCurrentSetting = m_camera.get_current_setting_number();
                Console.WriteLine("Save Setting to Disk!");
                Console.WriteLine("\nDo you want to save the current setting ({0}) as native({1})- or XML({2})-format? ", nCurrentSetting, Convert.ToUInt16(CS_SETTINGS_FORMAT.FORMAT_BINARY), Convert.ToUInt16(CS_SETTINGS_FORMAT.FORMAT_XML));
                Int16 nSettingFormat = Convert.ToInt16(Console.ReadLine());
                Console.WriteLine("\nEnter the setting number. Save as current setting(No.:{0}) or enter another number (Min: {1}- Max:{2}).", nCurrentSetting, CS_API.MIN_SETTING_NO, CS_API.MAX_SETTING_NO);
                ushort newSettingNo = Convert.ToUInt16(Console.ReadLine());
                String fileName;
                Console.WriteLine("Please enter the filename to save the setting to:");
                fileName = Console.ReadLine();
                if (fileName.LastIndexOf('.') < 0)
                {
                    if (nSettingFormat == Convert.ToInt16(CS_SETTINGS_FORMAT.FORMAT_BINARY))
                        fileName += ".mk";
                    else
                        fileName += ".set";
                }
                short nReturn = m_camera.save_current_setting_2Disk(fileName, nSettingFormat, newSettingNo );
                if(nReturn != CS_API.CS_OK)
                {
                    printAPIError(nReturn);
                }
                else{
                    Console.WriteLine("Successfully written fileName to disk!");
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Function will get the description of the currently active setting from the camera
        //////////////////////////////////////////////////////////////////////////
        public static void getSettingDescription()
        {
            if (m_isConnected)
            {
                StringBuilder description = new StringBuilder("", CS_API.MAX_SETTING_COMMENT_LENGTH);
//                string description = "This is an empty string";

                m_camera.get_setting_description(description,true);
                Console.WriteLine("The current setting description:\n ");
                Console.WriteLine(description);
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Function will set the setting description into the camera
        //////////////////////////////////////////////////////////////////////////
        public static void setSettingDescription()
        {
            if (m_isConnected)
            {
                Console.WriteLine("Please enter the setting comment!");

//                string description = Console.ReadLine();
                StringBuilder description = new StringBuilder(Console.ReadLine(), CS_API.MAX_SETTING_COMMENT_LENGTH); 
                m_camera.set_setting_description(description, true);

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the parameters used for the active references
        //////////////////////////////////////////////////////////////////////////
        public static void getReferenceSettings()
        {
            CS_REFERENCE_PARAMETER_STRUCT RefParamStruct = new CS_REFERENCE_PARAMETER_STRUCT();

            if (m_isConnected)
            {
                short nReturn = m_camera.get_reference_settings(RefParamStruct);
                if (nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\n");
                    Console.WriteLine("\n\nbUseBlackLevelCorrection: {0}", RefParamStruct.bUseBlackLevelCorrection);
                    Console.WriteLine("\nbUseShadingCorrection: {0}", RefParamStruct.bUseShadingCorrection);
                    Console.WriteLine("\nnBlackLevelReferenceNo: {0}", RefParamStruct.nBlackLevelReferenceNo);
                    Console.WriteLine("\nnShadingReferenceNo: {0}", RefParamStruct.nShadingReferenceNo);
                }
                else
                {
                    Console.WriteLine("Error getting referenced settings");
                    printAPIError(nReturn);
                }

            }
            else
                Console.WriteLine("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the parameters used for the active references
        //////////////////////////////////////////////////////////////////////////
        public static void setReferenceSettings()
        {
            if (m_isConnected)
            {
                CS_REFERENCE_PARAMETER_STRUCT RefParamStruct = new CS_REFERENCE_PARAMETER_STRUCT();

                Console.WriteLine("\n\nbUseBlackLevelCorrection (0=no or 1=yes): ");
                RefParamStruct.bUseBlackLevelCorrection = (Convert.ToUInt16(Console.ReadLine()) != 0);

                Console.WriteLine("bUseShadingCorrection (0=no or 1=yes): ");
                RefParamStruct.bUseShadingCorrection = (Convert.ToUInt16(Console.ReadLine()) != 0);

                Console.WriteLine("nBlackLevelReferenceNo: ");
                RefParamStruct.nBlackLevelReferenceNo = Convert.ToUInt16(Console.ReadLine());

                Console.WriteLine("nShadingReferenceNo: ");
                RefParamStruct.nShadingReferenceNo = Convert.ToUInt16(Console.ReadLine());

                short nError = m_camera.set_reference_settings(RefParamStruct);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting the reference settings in the camera \n");
                    printAPIError(nError);
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");

        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the camera into a mode where a valid image for generating the appropriate reference can  be acquired 
        //////////////////////////////////////////////////////////////////////////
        public static void prepareCam4Reference()
        {
            if (m_isConnected)
            {
                Console.WriteLine("\n\nEnter kind of reference ({0} = offset reference, {1} = shading reference): ", Convert.ToUInt16(CS_REFERENCE_MODE.BLACK_LEVEL_CORRECTION)
                    , Convert.ToUInt16(CS_REFERENCE_MODE.SHADING_CORRECTION));

                short nError = m_camera.prepare_cam_4reference((CS_REFERENCE_MODE)(Convert.ToUInt16(Console.ReadLine())));
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error preparing Camera for Reference!\n");
                    printAPIError(nError);
                }        

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// A reference image will be loaded into the API and valid reference data will be computed from this image 
        //////////////////////////////////////////////////////////////////////////
        public unsafe static void calculateReference()
        {
//             Console.WriteLine("\nFunction is currently not supported");
//             return;
            if (m_isConnected)
            {
                short tmp;
                UInt32 tmpU32=0;
		        CS_CALCULATE_REFERENCE_STRUCT calcRefParams = new CS_CALCULATE_REFERENCE_STRUCT();
		        Console.WriteLine("\n\nEnter the parameters for the reference calculation:");
		        Console.WriteLine("\nWhat kind of reference do you want to generate \n(0 = offset reference, 1 = shading reference)?\n\n");
                tmp = Convert.ToInt16(Console.ReadLine());
		        if(tmp <= 0)
			        tmp = 0;
		        if(tmp > (short)CS_REFERENCE_MODE.SHADING_CORRECTION)
		        {
			        Console.WriteLine("Wrong parameter entered, only 0 or 1 is valid !");
			        return;
		        }
		        calcRefParams.referenceType = (CS_REFERENCE_MODE)tmp;

		        // Check if the given image can be loaded for reference generation
		        bool bTry2Load = true;
                Bitmap  referenceImage = null;
                string fileName;

		        while(bTry2Load)
		        {
			        Console.WriteLine("\nEnter the path and name for the reference image:\n");
                    fileName = Console.ReadLine();
			        if(fileName.Length <= 0)
			        {
				        Console.WriteLine("Error entering file !")	;
			        }
			        else{
				        // Try to load the image
                        try { 
//                            CImage
                            //referenceImage = Image.FromFile(fileName);
                            
                            referenceImage = new Bitmap(fileName, false);
					        Console.WriteLine("\nSuccessfully loaded image with height:{0} width {1} \n", referenceImage.Height, referenceImage.Width);
					        bTry2Load = false;
				        }
                        catch (FileNotFoundException e)
				        {
					        Console.WriteLine("Loading of the image failed(File not found: {0}), do you want to try again (y/n)?", e.Message);
					        int choice = _getch();
					        if(choice != 'y')
						        return;
                        }
                        catch (OutOfMemoryException e)
				        {
                            Console.WriteLine("Loading of the image failed(invalid format: {0}), do you want to try again (y/n)?", e.Message);
					        int choice = _getch();
					        if(choice != 'y')
						        return;
				        }
                        catch (ArgumentException e)
				        {
                            Console.WriteLine("Loading of the image failed(Filename not valid: {0}), do you want to try again (y/n)?", e.Message);
					        int choice = _getch();
					        if(choice != 'y')
						        return;
                        }
			        }
		        }
                //BitmapData bmpData = referenceImage.LockBits(new Rectangle(0, 0, referenceImage.Width, referenceImage.Height),ImageLockMode.ReadWrite, PixelFormat.Format24bppRgb);
		        calcRefParams.width = (uint)referenceImage.Width ;
		        calcRefParams.height = (uint)referenceImage.Height;

                if (referenceImage.PixelFormat == System.Drawing.Imaging.PixelFormat.Format24bppRgb)
                    calcRefParams.pixelPitch = 3;
                else if (referenceImage.PixelFormat == System.Drawing.Imaging.PixelFormat.Format8bppIndexed)
                    calcRefParams.pixelPitch = 1;
                else
                {
                    Console.WriteLine("Error! Image format will not fit! Please provide a RGB-image with 3x8Bit color information or a grey image with 1x8Bit information");
                    return;
                }
///////////////////////////////////////////////////////////////////////////////////////////
// 2DO!!!!!!!!
// Did not find a correct method yet to get the real line pitch of the image
// I do assume here that there are no padding bytes in the image structure.
                calcRefParams.linePitch = referenceImage.Width * calcRefParams.pixelPitch;//bmpData.Stride;// *2; // Multiply by two to indicate that this is an int-Ptr
///////////////////////////////////////////////////////////////////////////////////////////
		        if(calcRefParams.pixelPitch < 8)
			        calcRefParams.pixelPitch = 1;
		        calcRefParams.pixelFormat =  (ushort) CS_PIXEL_FORMAT.RGB; // Only RGB possible when using C#-loading of images 
                BitmapData bmpData = referenceImage.LockBits(new Rectangle(0, 0, referenceImage.Width, referenceImage.Height), ImageLockMode.ReadWrite, referenceImage.PixelFormat);
//                BitmapData bdataRawA = m_rawImgA.LockBits(new System.Drawing.Rectangle(0, 0, m_rawImgA.Width, m_rawImgA.Height), System.Drawing.Imaging.ImageLockMode.WriteOnly, System.Drawing.Imaging.PixelFormat.Format24bppRgb);
                IntPtr pixelRaw = bmpData.Scan0;
                UInt64 srcImgSize = Convert.ToUInt64(referenceImage.Height) * Convert.ToUInt64(referenceImage.Width) * Convert.ToUInt64(calcRefParams.pixelPitch);

//                m_ptr_imgA = (byte*)Marshal.AllocHGlobal(Convert.ToInt32(srcImgSize)).ToPointer();
                byte[] bA = new byte[srcImgSize];
                Marshal.Copy(pixelRaw, bA, 0, Convert.ToInt32(srcImgSize));
                referenceImage.UnlockBits(bmpData);

///////////////////////////////////////////////////////////////////////////////
// 2DO!!!!!!!!
                ImageConverter converter = new ImageConverter();
                byte[] imgData = (byte[])converter.ConvertTo(referenceImage, typeof(byte[]));

                calcRefParams.pImage = bA;// imgData;// (SWIGTYPE_p_unsigned_char)bmpData.Scan0;
//////////////////////////////////////////////////////////////////////////////

		        Console.WriteLine("Enter the ID of the reference (0-3):\n");
                tmp = Convert.ToInt16(Console.ReadLine());
		        if(tmp <= 0)
			        tmp = 0;
		        if(tmp < 0 || tmp > 3)
		        {
			        Console.WriteLine("Wrong number entered, setting the reference ID to 0!");
			        tmp = 0;
		        }
		        calcRefParams.referenceNo = (ushort)tmp ;

                Console.WriteLine("Target value in 8 bit (default 255):\n");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp < 50 || tmp > 255)
                {
                    Console.WriteLine("Wrong number entered, setting the default value(255)!");
                    tmp = 0;
                }
                calcRefParams.targetValue = (ushort)tmp;

                if (m_camera.determineCameraType(false) == (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                {
                    calcRefParams.referenceVersion = CS_REFERENCE_VERSION.CS_ALLPIXA_WAVE_VERSION;


                    Console.WriteLine("Enter the ID of the color plane (Only important if this is a single color image):\n");
                    Console.WriteLine("Red plane:\t%d\n", CS_REFERENCE_PLANE_DEFINITION.RED_REF_PLANE);
                    Console.WriteLine("Green plane:\t%d\n", CS_REFERENCE_PLANE_DEFINITION.GREEN_REF_PLANE);
                    Console.WriteLine("Blue plane:\t%d\n", CS_REFERENCE_PLANE_DEFINITION.BLUE_REF_PLANE);
                    Console.WriteLine("Monochrome plane:\t%d\n", CS_REFERENCE_PLANE_DEFINITION.MONOCHROME_REF_PLANE);
                    tmp = Convert.ToInt16(Console.ReadLine());
                    if (tmp <= 0)
                        tmp = 0;
                    if (tmp < 0 || tmp > 4)
                    {
                        Console.WriteLine("Wrong number entered, setting the reference ID to MONOCHROME_PLANE!");
                        tmp = (short)CS_REFERENCE_PLANE_DEFINITION.MONOCHROME_REF_PLANE;
                    }
                    calcRefParams.planeType= (ushort)tmp;

                    // Get information regarding the set ROIs
                    Console.WriteLine("\nDo you want to enter the ROI settings manually (0=no,1=yes)? \n");
                    tmp = Convert.ToInt16(Console.ReadLine());
                    if (tmp <= 0)
                        tmp = 0;
                    if (tmp != 1)
                    {
                        // Retrieve ROI-Information from the camera
                        CS_ROI_PARAMETER_STRUCT roiParams = new CS_ROI_PARAMETER_STRUCT();
                        if (m_camera.get_roi_parameters(roiParams, false) != CS_API.CS_OK)
                        {
                            Console.WriteLine("Error retrieving ROI parameters from camera!\nTry setting this manually.");
                            return;
                        }
                        calcRefParams.roiParameters = roiParams;
                    }
                    else
                    {
                        // Ask user to enter the required ROI-information
                        int noOfROIs = -1;
                        while ((noOfROIs < 0) || (noOfROIs > CS_API.MAX_ROIS))
                        {
                            Console.WriteLine("Enter the number of ROIs(max. 4):\n");
                            noOfROIs = Convert.ToInt16(Console.ReadLine());
                            if ((noOfROIs < 0) || (noOfROIs > CS_API.MAX_ROIS))
                                tmp = 0;
                            if (tmp < 0 || tmp > CS_API.MAX_ROIS)
                            {
                                Console.WriteLine("Wrong number entered, setting the reference ID to 0!");
                                tmp = 0;
                            }
                        }
                        // Fill the ROIs
                        for (int i = 0; i < noOfROIs; i++)
                        {
                            calcRefParams.roiParameters.set_active(i, true);
                            // Ask for start position and width
                            Console.WriteLine("Start of ROI No. %d\n", i + 1);
                            tmpU32 = Convert.ToUInt32(Console.ReadLine());
                            if (tmp <= 0)
                                tmp = 1;
                            calcRefParams.roiParameters.set_start(i, tmpU32);
                            Console.WriteLine("Width of ROI No. %d\n", i + 1);
                            tmpU32 = Convert.ToUInt32(Console.ReadLine());
                            if (tmpU32 <= 0)
                                tmp = 1;
                            calcRefParams.roiParameters.set_width(i,tmpU32);

                        }
                        // Disable all other ROIs
                        for (int i = noOfROIs; i < CS_API.MAX_ROIS; i++)
                            calcRefParams.roiParameters.set_active(i, false);
                    }

                }    
            
                Console.WriteLine("Enter the path and name where the reference-file should be saved to:\nIf you do not want to save the file press enter.\n");
                fileName = Console.ReadLine(); 
		        if(fileName.Length == 0)
		        {
			        //Console.WriteLine("Error entering file !")	;
// 			        calcRefParams.pFileName = NULL;
		        }
		        else
		        {
			        if(fileName.Length > 0)
				        calcRefParams.pFileName = fileName;
			        else
				        calcRefParams.pFileName = "";
		        }


		        Console.WriteLine("Do you want to send the reference to the camera (0=no,1=yes)?\n");
                tmp = Convert.ToInt16(Console.ReadLine());
		        if(tmp <= 0)
			        tmp = 0;
		        calcRefParams.bSend2Camera = tmp != 0;
        //////////////////////////////////////////////////////////////////////////
        // Automatic calculation of the reference settings is not implemented yet
        // Will be realized in the future, please set the parameter manually!
        //////////////////////////////////////////////////////////////////////////
        // 		Console.WriteLine("\nDo you want to calculate the reference automatically (0=no,1=yes)?\n");
        // 		scanf_s("%d", &tmp);
        // 		if(tmp == 1 )
        // 			tmp = AUTOMATIC_DETECTION;
        // 		else
        // 			tmp = MANUAL_SETTING;
        // 		calcRefParams.calculationMode = (CS_CALC_REFERENCE_MODE)tmp;
		        calcRefParams.calculationMode = CS_CALC_REFERENCE_MODE.MANUAL_SETTING;
		        //////////////////////////////////////////////////////////////////////////
		        // Manual setting of the reference area
		        //////////////////////////////////////////////////////////////////////////
		        if(calcRefParams.calculationMode != CS_CALC_REFERENCE_MODE.AUTOMATIC_DETECTION)
		        {
			        Console.WriteLine("Enter the number of lines for the reference calculation\n");
                    tmp = Convert.ToInt16(Console.ReadLine());
			        if(tmp <= 0)
				        tmp = 0;
			        calcRefParams.numberOfLines = (ushort) tmp ;
			        Console.WriteLine("Enter the first line for the calculation:\n");
			        tmp = Convert.ToInt16(Console.ReadLine());
                    if(tmp <= 0)
				        tmp = 0;
			        calcRefParams.topLine = (ushort)tmp ;
			        Console.WriteLine("do you want to use extrapolation (0=no,1=yes)?\n");
                    tmp = Convert.ToInt16(Console.ReadLine());
			        if(tmp <= 0)
				        tmp = 0;
			        calcRefParams.doExtrapolation = Convert.ToBoolean(tmp);

			        //////////////////////////////////////////////////////////////////////////
			        // Get the parameters for the extrapolation
			        //////////////////////////////////////////////////////////////////////////
			        if(calcRefParams.doExtrapolation )
			        {
				        Console.WriteLine("Start of the extrapolation left\n");
                        tmp = Convert.ToInt16(Console.ReadLine());
                        if(tmp <= 0)
	    			        tmp = 0;
				        calcRefParams.leftExtrapolationStart = (ushort)tmp ;
				        Console.WriteLine("Start of the extrapolation right\n");
                        tmp = Convert.ToInt16(Console.ReadLine());
                        if(tmp <= 0)
	    			        tmp = 0;
				        calcRefParams.rightExtrapolationStart = (ushort)tmp ;
                        tmp = Convert.ToInt16(Console.ReadLine());
				        Console.WriteLine("Number of columns for the extrapolation ?");
                        if(tmp <= 0)
	    			        tmp = 0;
				        calcRefParams.numberOfColumnsForExtrapolation = (ushort)tmp ;
                        tmp = Convert.ToInt16(Console.ReadLine());
			        }
		        }

		        if(m_camera.calculate_reference(calcRefParams) != CS_API.CS_OK)
		        {
			        // Error calculating the reference
			        Console.WriteLine("\nError calculating the reference!");
		        }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// This will trigger the camera to create a reference by the incoming image data
        //////////////////////////////////////////////////////////////////////////
        public static void createReferenceInternally()
        {
            if (m_isConnected)
            {
                UInt16 referenceNo = 0;
                short referenceType = (short)CS_REFERENCE_MODE.BLACK_LEVEL_CORRECTION;
                UInt16 timeout = 60; // 60 seconds timeout

                Console.WriteLine("\n\nInternal reference generation");
                Console.WriteLine("\nMake sure that the camera is set into a valid working point and the correct target/light conditions are present.");
                Console.WriteLine("\nEnter reference number to store to (0 or 1): ");
                referenceNo = Convert.ToUInt16(Console.ReadLine());
                if (referenceNo <= 0)
                    referenceNo = 0;
                Console.WriteLine("Enter reference type to create (black level ref.: {0}, or shading ref.: {1}) ", Convert.ToUInt16(CS_REFERENCE_MODE.BLACK_LEVEL_CORRECTION), Convert.ToUInt16(CS_REFERENCE_MODE.SHADING_CORRECTION));
                referenceType = Convert.ToInt16(Console.ReadLine());
                if (referenceNo <= 0)
                    referenceNo = 0;

                if (m_camera.create_reference_internally((CS_REFERENCE_MODE)referenceType, referenceNo, timeout) == CS_API.CS_OK)
                {
                    // Reference successfully created
                    Console.WriteLine("Reference successfully created!\n");
                }
                else
                {
                    Console.WriteLine("\nError getting operation values from camera");
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the current white reference values of the allPIXA m_camera
        //////////////////////////////////////////////////////////////////////////
        public static void getCurrentWhiteReferenceValues()
        {
            if (m_isConnected)
            {
                // get the available number of gain values
                CS_CHANNEL_STRUCT whiteReferenceValues = new CS_CHANNEL_STRUCT();
                if (m_camera.get_current_reference_values(whiteReferenceValues, true) == CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nThe values of the current white references:\n");
                    for (int i = 0; i < whiteReferenceValues.no_of_valid_entries; i++)
                    {
                        Console.WriteLine("Element {0}: {1}", i, whiteReferenceValues.get_value(i));
                    }
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the white balancing parameters of the allPIXA camera
        //////////////////////////////////////////////////////////////////////////
        public static void getCameraWhiteBalancingSetup()
        {

            if (m_isConnected)
            {
		        CS_WHITE_BALANCE_STRUCT whiteBalanceParams = new CS_WHITE_BALANCE_STRUCT();

		        short nReturn = m_camera.get_camera_white_balancing_setup(whiteBalanceParams, true);
		        if(nReturn == CS_API.CS_OK)
		        {
			        Console.WriteLine("\n\nWhite balancing parameters:");
			        Console.WriteLine("White balancing On:\t\t{0}", whiteBalanceParams.enable_white_balancing);
                    Console.WriteLine("Target Value(red):\t\t{0}", whiteBalanceParams.red_target_value);
                    Console.WriteLine("Target Value(green):\t\t{0}", whiteBalanceParams.green_target_value);
                    Console.WriteLine("Target Value(blue):\t\t{0}", whiteBalanceParams.blue_target_value);
                    if (whiteBalanceParams.white_target_value != (short)CS_API.PARAMETER_NOT_USED)
                    {
                        Console.WriteLine("Target Value(white):\t\t{0}", whiteBalanceParams.white_target_value);
                    }
                    Console.WriteLine("Frames for average:\t\t{0}", (int)(System.Math.Pow(2.0, whiteBalanceParams.frames_for_average)));
                    Console.WriteLine("Use stop gain control:\t\t{0}", whiteBalanceParams.stopGainControl);
                    Console.WriteLine("Stop gain control factor:\t{0}", whiteBalanceParams.gain_control_stop_factor);
			        string text;

                    if (whiteBalanceParams.multi_tap_mode != CS_API.PARAMETER_NOT_USED)
                    {
                        switch (whiteBalanceParams.multi_tap_mode)
                        {
                            case (short)CS_MULTI_TAP_MODE.FRONT_TAP_IS_MASTER:
                                text = "Front tap is master";
                                break;
                            case (short)CS_MULTI_TAP_MODE.REAR_TAP_IS_MASTER:
                                text = "Rear tap is master";
                                break;
                            case (short)CS_MULTI_TAP_MODE.INDEPENDENT_WHITE_BALANCING:
                                text = "Tap independent white balancing";
                                break;
                            case (short)CS_MULTI_TAP_MODE.AUTOMATIC_MASTER_DETECTION:
                                text = "Automatic master detection";
                                break;
                            default:
                                text = "Unknown tap balancing mode";
                                break;
                        }
                        Console.WriteLine("Multi tap mode:\t\t\t{0}", text);
                    }
			        switch(whiteBalanceParams.white_balancing_mode)
			        {
			        case (short)CS_WHITE_BALANCING_MODE.GAIN_CONTROL_USING_AREA_RANGE:
				        text = "Gain control using area range";
				        break;
                    case (short)CS_WHITE_BALANCING_MODE.USE_MAXIMUM_VIDEO_LEVEL:
				        text = "Gain control using maximum video level in line";
				        break;
                    case (short)CS_WHITE_BALANCING_MODE.ODD_EVEN_CORRECTION_BY_LINE_AVG:
				        text = "Gain control using odd even correction";
				        break;
			        default:
				        text = "Unknown gain control mode";
				        break;
			        }
			        Console.WriteLine("Gain control mode:\t\t{0}", text);
                    Console.WriteLine("Synchronize with frame:\t\t{0}", whiteBalanceParams.synchronizeWithFrame);
                    Console.WriteLine("Show white reference borders:\t{0}", whiteBalanceParams.showWhiteReferenceBorders);
                     Console.WriteLine("Top:\t\t\t\t{0}", whiteBalanceParams.white_reference_area.top);
                     Console.WriteLine("Botom:\t\t\t\t{0}", whiteBalanceParams.white_reference_area.bottom);
                     Console.WriteLine("Left:\t\t\t\t{0}", whiteBalanceParams.white_reference_area.left);
                     Console.WriteLine("Right:\t\t\t\t{0}", whiteBalanceParams.white_reference_area.right);
                }
                else
                {
                    Console.WriteLine("Error getting white balancing setup!");
                    printAPIError(nReturn);
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }
        //////////////////////////////////////////////////////////////////////////
        /// Set the white balancing parameters of the allPIXA camera
        //////////////////////////////////////////////////////////////////////////
        public static void setCameraWhiteBalancingSetup()
        {

            if (m_isConnected)
            {
                short tmp;
                // get the available number of gain values
                CS_WHITE_BALANCE_STRUCT whiteBalancingSetting = new CS_WHITE_BALANCE_STRUCT();

                Console.WriteLine("\n\nPlease select if the white balancing should be turned on (0 or 1): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.enable_white_balancing = (tmp != 0);

                Console.WriteLine("Please enter the white reference target value(red): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.red_target_value = tmp;
                Console.WriteLine("Please enter the white reference target value(green): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.green_target_value = tmp;
                Console.WriteLine("Please enter the white reference target value(blue): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.blue_target_value = tmp;

                // The allPIXA wave camera does provide 4 sensor lines! Here set the parameters for the white(grey) sensor line
                if (m_camera.determineCameraType(false) == (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                {
                    Console.WriteLine("Please enter the white reference target value(white): ");
                    tmp = Convert.ToInt16(Console.ReadLine());
                    if (tmp <= 0)
                        tmp = 0;
                    whiteBalancingSetting.white_target_value = tmp;
                }

                Console.WriteLine("Enter the number of frames to average for the white reference(0, 2, 4, 8, 16 or 32): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.NO_AVERAGING;
                else
                {
                    switch (tmp)
                    {
                        case 0:
                            whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.NO_AVERAGING;
                            break;
                        case 2:
                            whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.USE_2_SAMPLES_FOR_AVERAGING;
                            break;
                        case 4:
                            whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.USE_4_SAMPLES_FOR_AVERAGING;
                            break;
                        case 8:
                            whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.USE_8_SAMPLES_FOR_AVERAGING;
                            break;
                        case 16:
                            whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.USE_16_SAMPLES_FOR_AVERAGING;
                            break;
                        case 32:
                            whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.USE_32_SAMPLES_FOR_AVERAGING;
                            break;
                        default:
                            whiteBalancingSetting.frames_for_average = (short)CS_AVERAGING.NO_AVERAGING;
                            break;
                    }
                }
                Console.WriteLine("Please select if the stop gain control should be used(0 or 1): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.stopGainControl = (tmp != 0);
                Console.WriteLine("Enter the stop gain control factor(double format): ");
                double fTmp = Convert.ToDouble(Console.ReadLine());
                if (fTmp <= 0.0)
                    fTmp = 0.0;
                whiteBalancingSetting.gain_control_stop_factor = (float)(fTmp + 0.0005);

                // allPIXA wave does only consist out of one tap!
                if (m_camera.determineCameraType(false) != (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                {
                    Console.WriteLine("Select which tap is used for white balancing. \n");
                    Console.WriteLine("0: Front tap is master\n");
                    Console.WriteLine("1: Rear tap is master\n");
                    Console.WriteLine("2: Independent white balancing\n");
                    Console.WriteLine("3: Automatic master detection\n");
                    tmp = Convert.ToInt16(Console.ReadLine());
                    if (tmp <= 0)
                        tmp = (short)CS_MULTI_TAP_MODE.AUTOMATIC_MASTER_DETECTION;
                    whiteBalancingSetting.multi_tap_mode = tmp;
                }

                Console.WriteLine("Should the white balancing be synchronized with the image frame(0/1): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.synchronizeWithFrame = (tmp != 0);

                Console.WriteLine("Enter the top position of the white balancing: ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.white_reference_area.top = tmp;
                Console.WriteLine("Enter the bottom position of the white balancing: ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.white_reference_area.bottom = tmp;
                Console.WriteLine("Enter the left position of the white balancing: ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.white_reference_area.left = tmp;
                Console.WriteLine("Enter the right position of the white balancing: ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.white_reference_area.right = tmp;

                Console.WriteLine("Display the white balancing borders in the image(0 or 1): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                whiteBalancingSetting.showWhiteReferenceBorders = Convert.ToBoolean(tmp);
                // Currently only one white balancing mode is possible, therefore we do not ask the value from the user
                whiteBalancingSetting.white_balancing_mode = (short)CS_WHITE_BALANCING_MODE.GAIN_CONTROL_USING_AREA_RANGE;

                short nError = m_camera.set_camera_white_balancing_setup(whiteBalancingSetting);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting the white reference target values: {0}\n", nError);
                    printAPIError(nError);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Check if the white balancing is running correct (The camera will perform a 
        /// white balancing with the currently active parameter)
        //////////////////////////////////////////////////////////////////////////
        public static void checkWhiteBalance()
        {
            if (m_isConnected)
            {
		        Console.WriteLine("\n\nChecking white balancing .....");
                short nReturn = m_camera.perform_white_balancing();
		        if(nReturn != CS_API.CS_OK)
		        {
                    Console.WriteLine("Error performing white balancing!");
                    printAPIError(nReturn);
                }
                else
                {
                    Console.WriteLine("White balancing finished OK!");
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Start a balancing between the two camera taps of the allPIXA camera
        //////////////////////////////////////////////////////////////////////////
        public static void doTapBalancing()
        {
            if (m_isConnected)
            {
		        Console.WriteLine("\n\nDoing tap balancing .....");
                short nReturn;
                nReturn = m_camera.do_tap_balancing();
		        if(nReturn  != CS_API.CS_OK)
		        {
			        Console.WriteLine("\nTap balancing failed!\n");
                    printAPIError(nReturn);
                }
                else
                {
                    Console.WriteLine("Tap balancing successful!");
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the currently active image parameters
        //////////////////////////////////////////////////////////////////////////
        public static void getImageParameter()
        {
            if (m_isConnected)
            {
                CS_IMAGE_PARAMETER_STRUCT imageParameter = new CS_IMAGE_PARAMETER_STRUCT();
                short nReturn;
                nReturn = m_camera.get_image_parameters(imageParameter, true);
                if (nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nFirst scan line delay: \t\t{0}", imageParameter.first_scan_line_delay);
                    Console.WriteLine("\nImage width: \t\t\t{0}", imageParameter.img_width);
                    Console.WriteLine("\nImage height: \t\t\t{0}", imageParameter.img_height);
                }
                else
                {
                    // Error getting the integration time settings from the camera
                    Console.WriteLine("Error getting the image parameter!");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the image parameters
        //////////////////////////////////////////////////////////////////////////
        public static void setImageParameter()
        {
            if (m_isConnected)
            {
                ushort tmp;
                CS_IMAGE_PARAMETER_STRUCT imageParameter = new CS_IMAGE_PARAMETER_STRUCT();
                Console.WriteLine("\n\nEnter the image parameters:");
                Console.WriteLine("\nFirst scan line delay:\n");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                imageParameter.first_scan_line_delay = tmp;
                Console.WriteLine("Image width:\n");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 100;
                imageParameter.img_width = tmp;
                Console.WriteLine("Image height:\n");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 100;
                imageParameter.img_height = tmp;
                short nError = m_camera.set_image_parameters(imageParameter);
                if (nError != CS_API.CS_OK)
                {
                    // Error setting the integration time parameters
                    Console.WriteLine("\n\nError setting the image parameter!\n");
                    printAPIError(nError);
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the currently active integration time and line period parameter
        //////////////////////////////////////////////////////////////////////////
        public static void getIntegrationTimeSettings()
        {
            if (m_isConnected)
            {
                CS_INTEGRATION_TIME_STRUCT integrationTime = new CS_INTEGRATION_TIME_STRUCT();
                short nReturn = m_camera.get_integration_time(integrationTime, true);
                if (nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nIntegration time in ns: {0}", integrationTime.integration_time_in_ns);
                    // allPIXA wave is using this feature by default, it is not possible to turn it off!
                    if(m_camera.determineCameraType(false) != (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                        Console.WriteLine("\nLine period feature: \t{0}", integrationTime.use_line_period_feature);
                    Console.WriteLine("\nLine period in ns: \t{0}", integrationTime.line_period_in_ns);
                }
                else
                {
                    // Error getting the integration time settings from the camera
                    Console.WriteLine("\n\nError getting the integration time!\n");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the integration time and line period parameter
        //////////////////////////////////////////////////////////////////////////
        public static void setIntegrationTimeSettings()
        {
            if (m_isConnected)
            {
                uint tmp;
                CS_INTEGRATION_TIME_STRUCT integrationTime= new CS_INTEGRATION_TIME_STRUCT();
                Console.WriteLine("\n\nEnter the integration time settings:");
                Console.WriteLine("\nIntegration time[ns]:\n");
                tmp = Convert.ToUInt32(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 100;
                integrationTime.integration_time_in_ns = tmp;
                if (m_camera.determineCameraType(false) != (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                {
                    Console.WriteLine("Line period feature (0 or 1):\n");
                    tmp = Convert.ToUInt32(Console.ReadLine());
                    if (tmp < 0)
                        tmp = 0;
                    integrationTime.use_line_period_feature = (tmp != 0);
                }
                else
                {
                    integrationTime.use_line_period_feature = true;
                }
                Console.WriteLine("Line period time[ns]:\n");
                tmp = Convert.ToUInt32(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 100;
                integrationTime.line_period_in_ns = tmp;
                short nReturn = m_camera.set_integration_time(integrationTime);
                if (nReturn != CS_API.CS_OK)
                {
                    // Error setting the integration time parameters
                    Console.WriteLine("\n\nError setting the integration time!\n");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the currently active properties for camLink connection
        //////////////////////////////////////////////////////////////////////////
        public static void getCamLinkProps()
        {
            if (m_isConnected)
            {
                CS_CAMLINK_PROPERTIES camLinkProps = new CS_CAMLINK_PROPERTIES();
                short nReturn = m_camera.get_camLink_properties(camLinkProps, true);
                if (nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nCamlink-Mode: {0}", camLinkProps.camLinkMode);
                    if (m_camera.determineCameraType(false) != (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                    {
                        switch (camLinkProps.camLinkMode)
                        {
                            case CS_CAMLINK_MODE.CL_BASE_MODE:
                                Console.WriteLine(" Base mode");
                                break;
                            case CS_CAMLINK_MODE.CL_MEDIUM_MODE:
                                Console.WriteLine(" Medium mode");
                                break;
                            case CS_CAMLINK_MODE.CL_MEDIUM_MODE_1X2:
                                Console.WriteLine(" Medium 1X2 mode");
                                break;
                            case CS_CAMLINK_MODE.CL_FULL64_1X8_RAW:
                                Console.WriteLine(" Full 64 1X8 raw mode");
                                break;
                            case CS_CAMLINK_MODE.CL_FULL80_1X8:
                                Console.WriteLine(" Full 80 1X8 mode");
                                break;
                            case CS_CAMLINK_MODE.CL_FULL80_1X10:
                                Console.WriteLine(" Full 80 1X10 mode");
                                break;
                            default:
                                Console.WriteLine(" Unknown CL-mode");
                                break;
                        }
                    }
                    else
                    {
                        switch (camLinkProps.camLinkMode)
                        {
                            case CS_CAMLINK_MODE.CL_BASE_MODE_WAVE:
                                Console.WriteLine(" Base mode");
                                break;
                            case CS_CAMLINK_MODE.CL_MEDIUM_1X4_MONO_WAVE:
                                Console.WriteLine(" Medium mono 1x4 mode");
                                break;
                            case CS_CAMLINK_MODE.CL_MEDIUM_1X4_4T_12BIT_MONO_WAVE:
                                Console.WriteLine(" Medium mono 1x4 12bit mode");
                                break;
                            case CS_CAMLINK_MODE.CL_MEDIUM_1X2_2T_8BIT_RGB_WAVE:
                                Console.WriteLine(" Medium rgb 1X2 mode");
                                break;
                            case CS_CAMLINK_MODE.CL_MEDIUM_1X4_1T_12BIT_RGB_WAVE:
                                Console.WriteLine(" Medium rgb 1X4 12bit mode");
                                break;
                            case CS_CAMLINK_MODE.CL_FULL64_1X8_8T_8BIT_RAW_WAVE:
                                Console.WriteLine(" Full 64 1X8 mode");
                                break;
                            case CS_CAMLINK_MODE.CL_FULL80_1X10_10T_8BIT_RAW_WAVE:
                                Console.WriteLine(" Full 80 1X10 8bit mode");
                                break;
                            case CS_CAMLINK_MODE.CL_FULL80_1X8_8T_10BIT_RAW_WAVE:
                                Console.WriteLine(" Full 80 1X8 mode");
                                break;
                            default:
                                Console.WriteLine(" Unknown CL-mode");
                                break;
                        }
                    }
                    Console.WriteLine("\nCamlink-Speed: {0}", camLinkProps.camLinkSpeed);
                }
                else
                {
                    Console.WriteLine("\n\nError getting CamLink properties from the camera");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the parameter for the camLink connection
        //////////////////////////////////////////////////////////////////////////
        public static void setCamLinkProps()
        {
            if (m_isConnected)
            {
                CS_CAMLINK_PROPERTIES camLinkProps = new CS_CAMLINK_PROPERTIES();
                short tmp;
                if(m_camera.determineCameraType(false) != (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                    Console.WriteLine("\n\nPlease select CamLink-Mode \n0 = Base\n1 = Medium (2XE geometry)\n2 = Medium (1X2 (allPIXA pro) geometry or 1x4 if allPIXA pro Mono is used)\n3 = Full 64 (1X8 geometry, only allPIXA pro)\n4 = Full 80 (1X8 geometry, only allPIXA pro)\n5 = Full 80 (1X10 geometry, only allPIXA pro)\n");
                else
                    Console.WriteLine("\n\nPlease select CamLink-Mode \n0 = Base\n1 = Medium mono (1x4 geometry)\n2 = Medium (1X4, 12bit geometry)\n3 = Medium (1X2 geometry)\n4 = Medium (1X4, 12bit geometry)\n5 = Full 64 raw(1X8 geometry)\n6 = Full 80 raw(1X10 geometry)\n7 = Full 80 raw(1X8 geometry)\n");

                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                camLinkProps.camLinkMode = (CS_CAMLINK_MODE)tmp;

                if (m_camera.determineCameraType(false) != (short)CS_CAMERA_TYPE.ALLPIXA_WAVE)
                {
                    Console.WriteLine("Please select CamLink-Speed (0=Standard, 1=HighSpeed or 2=ReducedSpeed): ");
                    tmp = Convert.ToInt16(Console.ReadLine());
                    if (tmp <= 0)
                        tmp = 0;
                    camLinkProps.camLinkSpeed = (CS_CAMLINK_SPEED)tmp;
                }
                short nError = m_camera.set_camLink_properties(camLinkProps);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting the cameraLink properties!");
                    printAPIError(nError);
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the currently active image processing parameters
        //////////////////////////////////////////////////////////////////////////
        public static void getImageProcessingParameter()
        {
            if (m_isConnected)
            {
                CS_IMAGE_PROCESSING_PARAMETER_STRUCT imageProcParams = new CS_IMAGE_PROCESSING_PARAMETER_STRUCT();
                short nReturn = m_camera.get_image_processing_parameters(imageProcParams, true);
                if (nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nMirror image: {0}", imageProcParams.mirror_image_horizontally);
                    Console.WriteLine("\nUse color conversion matrix: {0}", imageProcParams.use_color_conversion_matrix);
                    Console.WriteLine("\nUse linearisation table: {0}", imageProcParams.use_linearisation_table);
                    Console.WriteLine("\nUse keystone correction: {0}", imageProcParams.use_keystone_correction);
                    Console.WriteLine("\nKeystone pixel shift: {0}", imageProcParams.keystone_pixel_shift);
                    Console.WriteLine("\nKeystone correction width: {0}", imageProcParams.keystone_correction_width);
                    Console.WriteLine("\nSelect color conversion matrix: {0}", imageProcParams.select_color_conversion_matrix);
                }
                else
                {
                    Console.WriteLine("\n\nError getting brightness and contrast settings from camera");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the image processing parameters
        //////////////////////////////////////////////////////////////////////////
        public static void setImageProcessingParameter()
        {
            if (m_isConnected)
            {
                CS_IMAGE_PROCESSING_PARAMETER_STRUCT imageProcParams = new CS_IMAGE_PROCESSING_PARAMETER_STRUCT();
                short tmp;
                ushort utmp;
                Console.WriteLine("\n\nMirror Image (0=no or 1=yes): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                imageProcParams.mirror_image_horizontally = (0 != tmp);
                Console.WriteLine("Use color correction matrix (0=no or 1=yes): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                imageProcParams.use_color_conversion_matrix = (0 != tmp);
                Console.WriteLine("Use the linearisation table(0=no or 1=yes): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                imageProcParams.use_linearisation_table = (0 != tmp);
                Console.WriteLine("Use keystone correction (0=no or 1=yes): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                imageProcParams.use_keystone_correction = (0 != tmp);
                double dTmp;
                Console.WriteLine("Pixel shift for keystone correction: ");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                imageProcParams.keystone_pixel_shift = dTmp + 0.05;
                Console.WriteLine("Keystone correction correction width: ");
                utmp = Convert.ToUInt16(Console.ReadLine());
                imageProcParams.keystone_correction_width = utmp;
                Console.WriteLine("select color conversion matrix: ");
                utmp = Convert.ToUInt16(Console.ReadLine());
                if (utmp <= 1)
                    utmp = 1;
                imageProcParams.select_color_conversion_matrix = utmp;


                short nError = m_camera.set_image_processing_parameters(imageProcParams);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting the image processing parameters in the camera \n");
                    printAPIError(nError);
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the currently active physical setup parameters
        //////////////////////////////////////////////////////////////////////////
        public static void getPhysicalSetupParameter()
        {
            if (m_isConnected)
            {
                CS_PHYSICAL_SETUP_STRUCT physicalSetup = new CS_PHYSICAL_SETUP_STRUCT();
                short nReturn = m_camera.get_camera_physical_setup(physicalSetup, true);
                if ( nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nRGB line distance: {0}", physicalSetup.rgb_line_distance);
                    Console.WriteLine("\nScan direction: {0}", physicalSetup.scan_direction);
                }
                else
                {
                    Console.WriteLine("\n\nError getting physical setup parameter from camera");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the physical setup parameters
        //////////////////////////////////////////////////////////////////////////
        public static void setPhysicalSetupParameter()
        {
            if (m_isConnected)
            {
                CS_PHYSICAL_SETUP_STRUCT physicalSetup = new CS_PHYSICAL_SETUP_STRUCT();

                double dTmp;
                short tmp;

                Console.WriteLine("\n\nRBG line distance : ");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                physicalSetup.rgb_line_distance = (dTmp + 0.005);

                Console.WriteLine("Scan direction (0=forward or 1=backward): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                physicalSetup.scan_direction = Convert.ToUInt16(tmp);

                short nError = m_camera.set_camera_physical_setup(physicalSetup, true);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting physical setup parameters in the camera!\n");
                    printAPIError(nError);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the currently active video output parameters
        //////////////////////////////////////////////////////////////////////////
        public static void getVideoOutputParameter()
        {
            if (m_isConnected)
            {
		        CS_VIDEO_OUTPUT_STRUCT videoOutData = new CS_VIDEO_OUTPUT_STRUCT();
                short nReturn = m_camera.get_video_output_parameters(videoOutData, true);
		        if(nReturn == CS_API.CS_OK)
		        {
			        string outputMode;
			        switch(videoOutData.video_output_mode)
			        {
			        case (short)CS_VIDEO_OUTPUT_MODE._3x8_BIT_COLOR_PARALLEL:
				        outputMode = "3x8 Bit RGB in parallel(dual base)";
				        break;
                    case (short)CS_VIDEO_OUTPUT_MODE._2x8_BIT_GREY_FIRST_CL_PORT:
				        outputMode = "2x8 Bit grey on first CL port";
				        break;
                    case (short)CS_VIDEO_OUTPUT_MODE._2x10_BIT_GREY_FIRST_CL_PORT:
				        outputMode = "2*10 bit grey on first CL port";
				        break;
                    case (short)CS_VIDEO_OUTPUT_MODE._2x12_BIT_GREY_FIRST_CL_PORT:
				        outputMode = "2x12 bit grey on first CL port";
				        break;
                    case (short)CS_VIDEO_OUTPUT_MODE._2x8_BIT_GREY_CL_DUAL_BASE:
				        outputMode = "2x8 Bit grey on both CL ports(dual base)";
				        break;
                    case (short)CS_VIDEO_OUTPUT_MODE._2x10_BIT_GREY_CL_DUAL_BASE:
				        outputMode = "2x10 Bit grey on both CL ports(dual base)";
				        break;
                    case (short)CS_VIDEO_OUTPUT_MODE._2x12_BIT_GREY_CL_DUAL_BASE:
				        outputMode = "2x12 Bit grey on both CL ports(dual base)";
				        break;
			        default:
				        outputMode =  "Unknown video output format";
                        break;

			        }
			        Console.WriteLine("\n\nVideo output mode: {0}", outputMode);
                    Console.WriteLine("\nColor weight red: {0}", videoOutData.color_weight_red);
                    Console.WriteLine("\nColor weight green: {0}", videoOutData.color_weight_green);
                    Console.WriteLine("\nColor weight blue: {0}", videoOutData.color_weight_blue);
                    Console.WriteLine("\nSwap color channels (red and blue): {0}", videoOutData.swap_color_channels_red_blue);
                    Console.WriteLine("\nInsert mode: 0x{0:X}", videoOutData.insert_mode);
                    Console.WriteLine("\nposition each line data: {0}", videoOutData.position_eachline_data);
		        }
		        else
		        {
			        Console.WriteLine("\n\nError getting video output parameter from camera");
                    printAPIError(nReturn);
		        }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the video output parameters
        //////////////////////////////////////////////////////////////////////////
        public static void setVideoOutputParameter()
        {
            if (m_isConnected)
            {
                CS_VIDEO_OUTPUT_STRUCT videoOutData = new CS_VIDEO_OUTPUT_STRUCT();
                short tmp;

                Console.WriteLine("\n\nSet the video output mode: ");
                Console.WriteLine("\n{0:d} = RGB output: ", CS_VIDEO_OUTPUT_MODE._3x8_BIT_COLOR_PARALLEL);
                Console.WriteLine("\n{0:d} = 2x8 Bit grey on first CL port", CS_VIDEO_OUTPUT_MODE._2x8_BIT_GREY_FIRST_CL_PORT);
                Console.WriteLine("\n{0:d} = 2x10 Bit grey on first CL port", CS_VIDEO_OUTPUT_MODE._2x10_BIT_GREY_FIRST_CL_PORT);
                Console.WriteLine("\n{0:d} = 2x12 Bit grey on first CL port", CS_VIDEO_OUTPUT_MODE._2x12_BIT_GREY_FIRST_CL_PORT);
                Console.WriteLine("\n{0:d} = 2x8 Bit grey on both CL ports(dual base)", CS_VIDEO_OUTPUT_MODE._2x8_BIT_GREY_CL_DUAL_BASE);
                Console.WriteLine("\n{0:d} = 2x10 Bit grey on both CL ports(dual base)", CS_VIDEO_OUTPUT_MODE._2x10_BIT_GREY_CL_DUAL_BASE);
                Console.WriteLine("\n{0:d} = 2x12 Bit grey on both CL ports(dual base)\n", CS_VIDEO_OUTPUT_MODE._2x12_BIT_GREY_CL_DUAL_BASE);

                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                videoOutData.video_output_mode = tmp;
                double dTmp;

                Console.WriteLine("Color weight red: ");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                videoOutData.color_weight_red = (float)(dTmp + 0.005);
                Console.WriteLine("Color weight green: ");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                videoOutData.color_weight_green = (float)(dTmp + 0.005);
                Console.WriteLine("Color weight blue: ");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                videoOutData.color_weight_blue = (float)(dTmp + 0.005);

                Console.WriteLine("Swap color channels blue and red(0=no or 1=yes): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                videoOutData.swap_color_channels_red_blue = Convert.ToBoolean(tmp) ;

                Console.WriteLine("Insert mode (in HEX): ");
                string hexValue = Console.ReadLine(); 
                tmp = (short)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber); 
                if (tmp <= 0)
                    tmp = 0;
                videoOutData.insert_mode = tmp;

                Console.WriteLine("position each line data (0=start of line, 1=end of line, 2=start and end): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                videoOutData.position_eachline_data = tmp;

                short nError = m_camera.set_video_output_parameters(videoOutData);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting the video output parameters in the camera \n");
                    printAPIError(nError);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the parameters for the external triggering of the camera
        //////////////////////////////////////////////////////////////////////////
        public static void getTriggerSettings()
        {
            if (m_isConnected)
            {
		        CS_TRIGGER_STRUCT triggerParams = new CS_TRIGGER_STRUCT();
                short nReturn = m_camera.get_camera_trigger_mode(triggerParams, true);
		        if(nReturn == CS_API.CS_OK)
		        {
			        string text;
				
			        switch(triggerParams.trigger_mode)
			        {
			        case (short)CS_TRIGGER_MODE.FREE_RUNNING:
				        text = "Free running";
				        break;
                    case (short)CS_TRIGGER_MODE.START_CONDITION_ONLY:
				        text = "Use start condition";
				        break;
                    case (short)CS_TRIGGER_MODE.START_STOP_CONDITION:
				        text = "Use start and stop condition";
				        break;
			        default:
				        text = "Mode not defined";
                        break;
			        }
			        Console.WriteLine("\nTrigger mode: \t\t\t\t{0}", text);
                    Console.WriteLine("Scan lines after stop: \t\t\t{0}", triggerParams.scan_lines_after_stop);
                    Console.WriteLine("Stop after max. number of lines: \t{0}", triggerParams.stop_after_max_no_of_scan_lines);
                    Console.WriteLine("Max. number of scan lines: \t\t{0}", triggerParams.max_no_of_scan_lines);
                    Console.WriteLine("No. of suppressed lines: \t\t{0}", triggerParams.no_of_suppressed_lines);
			        switch(triggerParams.trigger_input)
			        {
			        case (short)CS_TRIGGER_INPUT.LIGHT_BARRIER_0:
				        text = "Light barrier 0";
				        break;
                    case (short)CS_TRIGGER_INPUT.LIGHT_BARRIER_1:
				        text = "Light barrier 1";
				        break;
                    case (short)CS_TRIGGER_INPUT.LIGHT_BARRIER_2:
				        text = "Light barrier 2";
				        break;
                    case (short)CS_TRIGGER_INPUT.LIGHT_BARRIER_3:
				        text = "Light barrier 3";
				        break;
			        default:
				        text = "Not defined";
				        break;
			        }
			        Console.WriteLine("Used trigger input: \t\t\t{0}", text);
			
			        switch(triggerParams.start_condition)
			        {
			        case (short)CS_TRIGGER_CONDITION.RAISING_EDGE:
				        text = "Raising edge";
				        break;
                    case (short)CS_TRIGGER_CONDITION.FALLING_EDGE:
				        text = "Falling edge";
				        break;
			        default:
				        text = "Not defined";
				        break;
			        }
			        Console.WriteLine("Start condition: \t\t\t{0}", text);
			        switch(triggerParams.stop_condition)
			        {
                        case (short)CS_TRIGGER_CONDITION.RAISING_EDGE:
				        text = "Raising edge";
				        break;
                        case (short)CS_TRIGGER_CONDITION.FALLING_EDGE:
				        text = "Falling edge";
				        break;
			        default:
				        text = "Not defined";
				        break;
			        }
			        Console.WriteLine("Stop condition: \t\t\t{0}", text);
			        switch(triggerParams.master_slave_operation)
			        {
			        case (short)CS_MASTER_SLAVE_OPERATION.NO_MASTER_SLAVE_OPERATION:
				        text = "No master slave operation";
				        break;
			        case (short)CS_MASTER_SLAVE_OPERATION.CAMERA_IS_MASTER:
				        text = "Camera is master";
				        break;
			        case (short)CS_MASTER_SLAVE_OPERATION.CAMERA_IS_SLAVE:
				        text = "Camera is slave";
				        break;
			        case (short)CS_MASTER_SLAVE_OPERATION.SELECT_MASTER_SLAVE_BY_INPUT:
				        text = "Master/Slave selected by GPIO";
				        break;
			        default:
				        text = "Not defined";
				        break;
			        }
                    Console.WriteLine("Master slave operation: \t\t{0}", text);
		        }
		        else
		        {
			        Console.WriteLine("\nError trigger settings from camera");
                    printAPIError(nReturn);
		        }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the parameters for the external triggering of the camera
        //////////////////////////////////////////////////////////////////////////
        public static void setTriggerSettings()
        {
            if (m_isConnected)
            {
                CS_TRIGGER_STRUCT triggerParams = new CS_TRIGGER_STRUCT();
                short tmp;

                Console.WriteLine("\nSet the trigger mode: \nFree running: \t\t\t{0:d}\nOnly start condition:\t\t{1:d}\nStart and stop condition:\t{2:d}\n"
                    , CS_TRIGGER_MODE.FREE_RUNNING, CS_TRIGGER_MODE.START_CONDITION_ONLY, CS_TRIGGER_MODE.START_STOP_CONDITION);
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = (short)CS_TRIGGER_MODE.FREE_RUNNING;
                triggerParams.trigger_mode = tmp;
                Console.WriteLine("\nScan lines after stop: ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.scan_lines_after_stop = (ushort)tmp;
                Console.WriteLine("\nStop after max. number of lines (0: do not stop, 1: stop): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.stop_after_max_no_of_scan_lines = (tmp != 0);
                Console.WriteLine("\nStop after max. number of lines (Count of lines): ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.max_no_of_scan_lines = (ushort)tmp;
                Console.WriteLine("No. of suppressed lines:\n");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.no_of_suppressed_lines = (ushort)tmp;
                Console.WriteLine("\nTrigger Input (Light barrier 0= {0:d}, LB1={1:d}, LB2={2:d} LB3={3:d})"
                    , CS_TRIGGER_INPUT.LIGHT_BARRIER_0, CS_TRIGGER_INPUT.LIGHT_BARRIER_1, CS_TRIGGER_INPUT.LIGHT_BARRIER_2, CS_TRIGGER_INPUT.LIGHT_BARRIER_3);
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.trigger_input = tmp;
                Console.WriteLine("\nStart Condition (Raising edge = {0:d}, falling edge = {1:d})"
                    , CS_TRIGGER_CONDITION.RAISING_EDGE, CS_TRIGGER_CONDITION.FALLING_EDGE);
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.start_condition = tmp;
                Console.WriteLine("\nStop Condition (Raising edge = {0:d}, falling edge = {1:d})", CS_TRIGGER_CONDITION.RAISING_EDGE, CS_TRIGGER_CONDITION.FALLING_EDGE);
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.stop_condition = tmp;
                Console.WriteLine("\nMaster/slave setting (No master slave operation= {0:d}\nCamera is master={1:d}\nCamera is slave={2:d}\nMaster slave selected by GPIO={3:d})\n"
                    , CS_MASTER_SLAVE_OPERATION.NO_MASTER_SLAVE_OPERATION, CS_MASTER_SLAVE_OPERATION.CAMERA_IS_MASTER, CS_MASTER_SLAVE_OPERATION.CAMERA_IS_SLAVE, CS_MASTER_SLAVE_OPERATION.SELECT_MASTER_SLAVE_BY_INPUT);
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                triggerParams.master_slave_operation = tmp;

                short nError = m_camera.set_camera_trigger_mode(triggerParams);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting the trigger mode in the camera!");
                    printAPIError(nError);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the parameters for the encoder mode of the camera
        //////////////////////////////////////////////////////////////////////////
        public static void getEncoderSettings()
        {
            if (m_isConnected)
            {
		        CS_ENCODER_STRUCT encoderSettings = new CS_ENCODER_STRUCT();
                short nReturn = m_camera.get_encoder_parameters(encoderSettings, true);
		        if(nReturn == CS_API.CS_OK)
		        {
			        Console.WriteLine("\n\nEncoder enabled: \t\t{0}", encoderSettings.enable_ecoder);
                    Console.WriteLine("\nIncrements per um: \t\t{0}", encoderSettings.um_per_increment);
                    Console.WriteLine("\nNumber of encoder channels: \t{0}", encoderSettings.encoder_channels);
                    Console.WriteLine("\nLine trigger reduction: \t{0}", encoderSettings.line_trigger_reduction);
                    Console.WriteLine("\nVertical resolution: \t\t{0} DPI", encoderSettings.vertical_resolution_dpi);
			        string buffer;
			        switch(encoderSettings.synchronisation_mode)
			        {
			        case (ushort)CS_ENCODER_SYNC_MODE.STANDARD_ENCODER:
				        buffer = "STANDARD_ENCODER";
				        break;
                    case (ushort)CS_ENCODER_SYNC_MODE.LINE_TRIGGER:
				        buffer = "LINE_TRIGGER";
				        break;
			        default:
				        buffer = "Unknown sync mode";
				        break;
			        }
                    Console.WriteLine("\nSynchronization mode: \t\t{0}", buffer);
                    Console.WriteLine("\nAveraging: \t\t\t{0}", (int)(System.Math.Pow(2.0, encoderSettings.averaging)));
		        }
		        else
		        {
			        Console.WriteLine("\n\nError getting encoder parameters from camera");
                    printAPIError(nReturn);
		        }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the parameters for the encoder mode of the camera
        //////////////////////////////////////////////////////////////////////////
        public static void setEncoderSettings()
        {
            if (m_isConnected)
            {
                CS_ENCODER_STRUCT encoderSettings = new CS_ENCODER_STRUCT();
                ushort tmp;
                double dTmp;

                Console.WriteLine("\n\nUse Encoder(0=no or 1=yes): ");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                encoderSettings.enable_ecoder = Convert.ToBoolean(tmp);

                Console.WriteLine("Increments per um:\n");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                encoderSettings.um_per_increment = dTmp + 0.0005;

                Console.WriteLine("\nNumber of encoder channels: ");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                encoderSettings.encoder_channels = tmp;

                Console.WriteLine("\nLine trigger reduction: ");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                encoderSettings.line_trigger_reduction = tmp;

                Console.WriteLine("\nVertical resolution in DPI: ");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                encoderSettings.vertical_resolution_dpi = dTmp + 0.0005;

                Console.WriteLine("\nSynchronisation mode (Standard encoder: {0:d}, Line Trigger: {1:d})"
                    , CS_ENCODER_SYNC_MODE.STANDARD_ENCODER, CS_ENCODER_SYNC_MODE.LINE_TRIGGER);
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                encoderSettings.synchronisation_mode = tmp;

                Console.WriteLine("Enter the number of encoder pulses to average (0, 2, 4, 8 or 16): ");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    encoderSettings.averaging = (ushort)CS_AVERAGING.NO_AVERAGING;
                else
                {
                    switch (tmp)
                    {
                        case 0:
                            encoderSettings.averaging = (ushort)CS_AVERAGING.NO_AVERAGING;
                            break;
                        case 2:
                            encoderSettings.averaging = (ushort)CS_AVERAGING.USE_2_SAMPLES_FOR_AVERAGING;
                            break;
                        case 4:
                            encoderSettings.averaging = (ushort)CS_AVERAGING.USE_4_SAMPLES_FOR_AVERAGING;
                            break;
                        case 8:
                            encoderSettings.averaging = (ushort)CS_AVERAGING.USE_8_SAMPLES_FOR_AVERAGING;
                            break;
                        case 16:
                            encoderSettings.averaging = (ushort)CS_AVERAGING.USE_16_SAMPLES_FOR_AVERAGING;
                            break;
                        default:
                            encoderSettings.averaging = (ushort)CS_AVERAGING.NO_AVERAGING;
                            break;
                    }
                }

                short nError = m_camera.set_encoder_parameters(encoderSettings, true);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting the encoder parameters in the camera \n");
                    printAPIError(nError);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the currently active IO configuration parameters
        //////////////////////////////////////////////////////////////////////////
        public static void getIOConfigurationParameter()
        {
            if (m_isConnected)
            {
                CS_IO_CONFIG_STRUCT IOConfig = new CS_IO_CONFIG_STRUCT();
                short nReturn = m_camera.get_IO_signal_configuration(IOConfig, true);
                if (nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nEncoder Config: 0x{0:X}", IOConfig.encoder_config);
                    Console.WriteLine("\nINCR0: 0x{0:X}", IOConfig.encoder_incr0);
                    Console.WriteLine("\nINCR1: 0x{0:X}", IOConfig.encoder_incr1);
                    Console.WriteLine("\nLight barrier: 0x{0:X}", IOConfig.light_barrier);
                    Console.WriteLine("\nGeneral function: 0x{0:X}", IOConfig.general_functions);
                }
                else
                {
                    Console.WriteLine("\n\nError getting IO Config parameter from camera!");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the IO configuration parameters
        //////////////////////////////////////////////////////////////////////////
        public static void setIOConfigurationParameter()
        {
            if (m_isConnected)
            {
                ushort tmp = 0; 
                CS_IO_CONFIG_STRUCT IOConfig = new CS_IO_CONFIG_STRUCT();

                Console.WriteLine("\n\nEncoder Config(enter as hex-value): ");
                string hexValue = Console.ReadLine();
                tmp = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tmp <= 0)
                    tmp = 0;

                IOConfig.encoder_config = tmp;

                Console.WriteLine("INCR0(enter as hex-value): ");
                hexValue = Console.ReadLine(); 
                tmp = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tmp <= 0)
                    tmp = 0;
                IOConfig.encoder_incr0 = tmp;

                Console.WriteLine("INCR1(enter as hex-value): ");
                hexValue = Console.ReadLine();
                tmp = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tmp <= 0)
                    tmp = 0;
                IOConfig.encoder_incr1 = tmp;

                Console.WriteLine("Light barrier(enter as hex-value): ");
                hexValue = Console.ReadLine();
                tmp = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tmp <= 0)
                    tmp = 0;
                IOConfig.light_barrier = tmp;

                Console.WriteLine("General functions(enter as hex-value): ");
                hexValue = Console.ReadLine();
                tmp = (ushort)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
                if (tmp <= 0)
                    tmp = 0;
                IOConfig.general_functions = tmp;

                short nError = m_camera.set_IO_signal_configuration(IOConfig);
                if (nError != CS_API.CS_OK)
                {
                    Console.WriteLine("Error setting IO configuration parameters in the camera \n");
                    printAPIError(nError);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Gets the parameters for the LED flash parameters
        //////////////////////////////////////////////////////////////////////////
        public static void getLedFlashSettings()
        {
            if (m_isConnected)
            {
		        CS_LED_FLASH_CONTROL_STRUCT ledFlashParams = new CS_LED_FLASH_CONTROL_STRUCT();
                short nReturn = m_camera.get_led_flash_parameters(ledFlashParams, true) ;
		        if(nReturn == CS_API.CS_OK)
		        {
			        Console.WriteLine("\nFlash mode enabled: {0}\n", ledFlashParams.enableFlashControl);
			        Console.WriteLine("Number of flash patterns: {0}\n", ledFlashParams.numberOfLinePatterns);
			        Console.WriteLine("Led flash sequence time: {0}\n", ledFlashParams.flashSequenceTime);
			
			        Console.WriteLine("\nPattern 1: \tPattern 2\tPattern 3\tPattern 4\n");
                    for (int i = 0; i < CS_LED_FLASH_CONTROL_STRUCT.MAX_LED_FLASH_PATTERNS; i++)
                    {
                        for (int iOutput = 0; iOutput < CS_LED_FLASH_CONTROL_STRUCT.MAX_OUTPUTS; iOutput++)
				        {
					        Console.WriteLine("Output {0}: {1}\t", iOutput + 1, ledFlashParams.get_flashDefinitions(i, iOutput));
				        }
				        Console.WriteLine(" ");
			        }
			        Console.WriteLine("LED flash synchronisation: {0}\n", ledFlashParams.ledDriverSync);
		        }
		        else
		        {
			        Console.WriteLine("\n\nError getting led flash parameter from camera");
                    printAPIError(nReturn);
		        }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sets the parameters for the LED flash parameters
        //////////////////////////////////////////////////////////////////////////
        public static void setLedFlashSettings()
        {
            if (m_isConnected)
            {
                CS_LED_FLASH_CONTROL_STRUCT ledFlashParams = new CS_LED_FLASH_CONTROL_STRUCT();

                ushort tmp = 0;
                double dTmp = 0.0;
                Console.WriteLine("Disable(0) or enable(1) LED flash: ");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                ledFlashParams.enableFlashControl = tmp;

                Console.WriteLine("\nNumber of flash-patterns(max. {0:d}): ", CS_LED_FLASH_CONTROL_STRUCT.MAX_LED_FLASH_PATTERNS);
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp < CS_LED_FLASH_CONTROL_STRUCT.MAX_LED_FLASH_PATTERNS)
                    ledFlashParams.numberOfLinePatterns = tmp;
                else
                {
                    Console.WriteLine("Flash pattern count out of bounds!\nSetting parameter to:{0:d}\n", CS_LED_FLASH_CONTROL_STRUCT.MAX_LED_FLASH_PATTERNS);
                }

                Console.WriteLine("LED flash sequence time in us: ");
                dTmp = Convert.ToDouble(Console.ReadLine());
                if (dTmp <= 0.0)
                    dTmp = 0.0;
                ledFlashParams.flashSequenceTime = dTmp;

                for (int i = 0; i < ledFlashParams.numberOfLinePatterns; i++)
                {
                    Console.WriteLine("\nPattern no: {0}\n", i);
                    for (int iOutput = 0; iOutput < CS_LED_FLASH_CONTROL_STRUCT.MAX_OUTPUTS; iOutput++)
                    {
                        Console.WriteLine("Time for Output {0} in us: ", iOutput);
                        dTmp = Convert.ToDouble(Console.ReadLine());
                        if (dTmp <= 0.0)
                            dTmp = 0.0;
                        ledFlashParams.set_flashDefinitions(i, iOutput, dTmp);
                    }
                    Console.WriteLine("\n");
                }

                Console.WriteLine("Disable(0) or enable(1) LED synchronisation clock: ");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                ledFlashParams.ledDriverSync = tmp;

                short nReturn = m_camera.set_led_flash_parameters(ledFlashParams, true);
                if (nReturn != CS_API.CS_OK)
                {
                    Console.WriteLine("\n\nError setting led flash parameter to camera");
                    printAPIError(nReturn);
                }
                else
                {
                    Console.WriteLine("Successfully set the LED flash parameters!");
                }


            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the currently active test pattern settings
        //////////////////////////////////////////////////////////////////////////
        public static void getTestPatternSettings()
        {
            if (m_isConnected)
            {
		        CS_TEST_PATTERN_STRUCT testPattern = new CS_TEST_PATTERN_STRUCT();
                short nReturn = m_camera.get_test_pattern_settings(testPattern, true);
		        if(nReturn == CS_API.CS_OK) 
		        {
			        string text;
			        switch(testPattern.nTestPattern)
			        {
			        case (ushort)CS_TEST_PATTERN.TEST_PATTERN_OFF:
				        text =  "Test patterns is switched off";
				        break;
			        case (ushort)CS_TEST_PATTERN.GREY_RAMP_IN_CCD_DIR:
				        text =  "Grey ramp in CCD direction";
				        break;
			        case (ushort)CS_TEST_PATTERN.GREY_RAMP_IN_TRANSPORT_DIR:
				        text =  "Grey ramp in transport direction";
				        break;
			        case (ushort)CS_TEST_PATTERN.INPUT_RAMP_ON_GREEN_CHANNEL:
				        text =  "Ramp in the green channel (red and blue set on test pattern level)";
				        break;
			        case (ushort)CS_TEST_PATTERN.SEQUENCE_OF_PATTERNS:
				        text =  "Alternating test patterns";
				        break;
			        case (ushort)CS_TEST_PATTERN.CHANGE_VIDEO_LEVEL_AT_EVERY_PIXEL:
				        text =  "Video levels are changed for each pixel";
				        break;
			        default:
				        text =  "Unknown test pattern";
				        break;
			        }
			        Console.WriteLine("\n\nTest pattern:\t{0}\n", text);
			        Console.WriteLine("Test pattern level:\t{0}\n", testPattern.nTestPatternLevel);
		        }
		        else
		        {
			        // Error getting the test pattern settings
			        Console.WriteLine("\n\nError getting the test pattern setting!");
                    printAPIError(nReturn);
		        }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the test pattern settings
        //////////////////////////////////////////////////////////////////////////
        public static void setTestPatternSettings()
        {
            if (m_isConnected)
            {
                CS_TEST_PATTERN_STRUCT testPattern = new CS_TEST_PATTERN_STRUCT();
                ushort tmp;
                Console.WriteLine("\n\nEnter the desired test pattern! \n");
                Console.WriteLine("\nTest pattern off: \t\t\t{0:d} ", CS_TEST_PATTERN.TEST_PATTERN_OFF);
                Console.WriteLine("\nGrey ramp in CCD direction: \t\t{0:d} ", CS_TEST_PATTERN.GREY_RAMP_IN_CCD_DIR);
                Console.WriteLine("\nGrey ramp in transport direction: \t{0:d} ", CS_TEST_PATTERN.GREY_RAMP_IN_TRANSPORT_DIR);
                Console.WriteLine("\nRamp in green channel: \t\t\t{0:d} ", CS_TEST_PATTERN.INPUT_RAMP_ON_GREEN_CHANNEL);
                Console.WriteLine("\nSequence of patterns: \t\t\t{0:d} ", CS_TEST_PATTERN.SEQUENCE_OF_PATTERNS);
                Console.WriteLine("\nChange video for every channel: \t{0:d} \n", CS_TEST_PATTERN.CHANGE_VIDEO_LEVEL_AT_EVERY_PIXEL);
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                testPattern.nTestPattern = tmp;

                Console.WriteLine("Enter the desired test pattern level:");
                tmp = Convert.ToUInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                testPattern.nTestPatternLevel = tmp;
                short nReturn = m_camera.set_test_pattern_settings(testPattern, true);
                if (nReturn != CS_API.CS_OK)
                {
                    // Error setting the test pattern value!
                    Console.WriteLine("\n\nError setting the test pattern!");
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the logging parameter of the CSAPI
        //////////////////////////////////////////////////////////////////////////
        public static void getLoggingParameter()
        {
            string fileName = "";
            CS_LOGGING_PARAM_STRUCT LogParam = new CS_LOGGING_PARAM_STRUCT();
            LogParam.pLogFileName = fileName;

            short nReturn = m_camera.get_logging_parameter (LogParam) ;
            if (nReturn == CS_API.CS_OK)
            {
                Console.WriteLine("\n\nLogFileName: {0}, Output to DbgViewer: {1}", LogParam.pLogFileName, LogParam.OutputToDbgViewer);
                Console.WriteLine("\nLoggingMask: 0x{0:x}, InterfaceLength: {1}, TimeStamp: {2}", LogParam.LoggingMask, LogParam.InterfaceDataLength, LogParam.TimeStamp);
            }
            else
            {
                Console.WriteLine("\n\nError getting logging parameter from camera");
                printAPIError(nReturn);
            }
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the logging parameter of the CSAPI
        //////////////////////////////////////////////////////////////////////////
        public static void setLoggingParameter()
        {
            uint tmp=0; // Buffer variable because scanf is using 32 Bit values!

            CS_LOGGING_PARAM_STRUCT LogParams = new CS_LOGGING_PARAM_STRUCT();
            bool bStoreToReg = false;
            string fileName = "";
            LogParams.pLogFileName = fileName;

            Console.WriteLine("\n\nEnter path and name for logging data (0=empty) : ");
            fileName = Console.ReadLine();
            if (fileName.Length <= 0)
            {
                Console.WriteLine("Error entering file !")	;
                fileName = "0";
            }
            else
            {
                if ( fileName == "0" ) 
                {
                    Console.WriteLine("empty fileName!\n")	;
                    fileName = "0";
                }
            }
            LogParams.pLogFileName = fileName;

            Console.WriteLine("Output data to DbgViewer(0=no or 1=yes): ");
            tmp = Convert.ToUInt16(Console.ReadLine());
            if (tmp <= 0)
                tmp = 0;
            LogParams.OutputToDbgViewer = tmp;

            Console.WriteLine("Set logging mask (hex format): ");
            string hexValue = Console.ReadLine();
            tmp = (uint)int.Parse(hexValue, System.Globalization.NumberStyles.HexNumber);
            if (tmp <= 0)
                tmp = 0;
            LogParams.LoggingMask = tmp;

            Console.WriteLine("Set Interface data length (0=off, 1=header, >1=complete): ");
            tmp = Convert.ToUInt16(Console.ReadLine());
            if (tmp <= 0)
                tmp = 0;
            LogParams.InterfaceDataLength = tmp;

            Console.WriteLine("Insert Time Stamp (0=no or 1=yes): ");
            tmp = Convert.ToUInt16(Console.ReadLine());
            if (tmp <= 0)
                tmp = 0;
            LogParams.TimeStamp = tmp;

            Console.WriteLine("Store to Registry(0=no or 1=yes): ");
            tmp = Convert.ToUInt16(Console.ReadLine());
            if (tmp <= 0)
                tmp = 0;
            bStoreToReg = (tmp != 0);

            short nError = m_camera.set_logging_parameter (LogParams, bStoreToReg);
            if( nError != CS_API.CS_OK)
            {
                Console.WriteLine("Error setting the logging parameters in the camera \n");
                printAPIError(nError);
            }
            else
            {
                Console.WriteLine("Logging parameters were set successfully!");
            }
        }


        //////////////////////////////////////////////////////////////////////////
        /// Get the currently active trace settings
        //////////////////////////////////////////////////////////////////////////
        public static void getTraceMode()
        {
            if (m_isConnected)
            {
                ushort traceMode = 0;
                short nReturn = m_camera.get_trace_mode(ref traceMode, true) ;
                if (nReturn == CS_API.CS_OK)
                {
                    Console.WriteLine("\nTrace modes:");
                    Console.WriteLine("General debug info \t\t{0}", (traceMode &  CS_API.TRACE_GENERAL_DEBUG_INFO) != 0);
                    Console.WriteLine("Transport layer \t\t{0}", (traceMode & CS_API.TRACE_TRANSPORT_LAYER) != 0);
                    Console.WriteLine("Detailed transport layer info:\t{0}", (traceMode & CS_API.TRACE_TRANSPORT_LAYER_DETAILS) != 0);
                    Console.WriteLine("White control \t\t\t{0}", (traceMode & CS_API.TRACE_WHITE_CONTROL) != 0);
                    Console.WriteLine("Environmental values \t\t{0}", (traceMode & CS_API.TRACE_ENVIRONMENT_VALUES) != 0);
                    Console.WriteLine("Image counter \t\t\t{0}", (traceMode & CS_API.TRACE_IMAGE_COUNTER) != 0);
                }
                else
                {
                    Console.WriteLine("\n\nWas not able to retrieve the current trace mode!\n");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Set the trace settings
        //////////////////////////////////////////////////////////////////////////
        public static void setTraceMode()
        {
            if (m_isConnected)
            {
                ushort traceMode = 0;
                short tmp;

                Console.WriteLine("\nSet the trace modes:");
                Console.WriteLine("\nGeneral debug info (0: no logging, 1: log into trace) ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp > 0)
                    traceMode = (ushort)(traceMode | CS_API.TRACE_GENERAL_DEBUG_INFO);
                Console.WriteLine("Transport layer (0: no logging, 1: log into trace) ");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp > 0)
                    traceMode = (ushort)(traceMode | CS_API.TRACE_TRANSPORT_LAYER);
                Console.WriteLine("Detailed transport layer info (0: no logging, 1: log into trace)");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp > 0)
                    traceMode = (ushort)(traceMode | CS_API.TRACE_TRANSPORT_LAYER_DETAILS);
                Console.WriteLine("White control (0: no logging, 1: log into trace):");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp > 0)
                    traceMode = (ushort)(traceMode | CS_API.TRACE_WHITE_CONTROL);
                Console.WriteLine("nEnvironmental values (0: no logging, 1: log into trace):");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp > 0)
                    traceMode = (ushort)(traceMode | CS_API.TRACE_ENVIRONMENT_VALUES);
                Console.WriteLine("Image counter (0: no logging, 1: log into trace):");
                tmp = Convert.ToInt16(Console.ReadLine());
                if (tmp <= 0)
                    tmp = 0;
                if (tmp > 0)
                    traceMode = (ushort)(traceMode | CS_API.TRACE_IMAGE_COUNTER);

                short nReturn = m_camera.set_trace_mode(traceMode, true) ;
                if (nReturn != CS_API.CS_OK)
                {
                    Console.WriteLine("\nError setting the trace mode in the camera!!!\n");
                    printAPIError(nReturn);
                }

            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Get the internal camera trace messages and display them. Single messages 
        /// are divided by a new line character
        //////////////////////////////////////////////////////////////////////////
        public static void getCameraTrace()
        {
            if (m_isConnected)
            {
         //       string traceMsg = "";
                short traceLength = 0;

////////////////////////////////////////////////////////////////////////////////////////////
                short nReturn = 0;
                StringBuilder cameraTrace = new StringBuilder("", CS_API.MAX_TRACE_MESSAGE_TEXT_LENGTH);

                nReturn = m_camera.get_trace_from_camera(cameraTrace, ref traceLength);
                if (nReturn == CS_API.CS_OK)
                {
                    if ((cameraTrace.Length > 0) && traceLength > 0)
                    {
//                        int dataOffset = 0;
                        Console.WriteLine("\nTrace from camera(Length: {0}):\n", traceLength);
                        Console.WriteLine(cameraTrace);
                    }
                }
                else
                {
                    Console.WriteLine("\n\nError getting trace from camera: {0}");
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// Sends a given HSI file to the camera
        //////////////////////////////////////////////////////////////////////////
        public static void storeDataToCameraMemory()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the file name to send(including path):\n ");
                string fileName = Console.ReadLine();
                if (fileName.Length > 0)
                {
                    byte[] buffer;
                    int sum = 0;                          // total number of bytes read
                    FileStream fileStream = new FileStream(fileName, FileMode.Open, FileAccess.Read);
                    try
                    {
                        int length = (int)fileStream.Length;  // get file length
                        buffer = new byte[length];            // create buffer
                        Int32 count;                            // actual number of bytes read

                        // read until Read method returns 0 (end of the stream has been reached)
                        while ((count = fileStream.Read(buffer, sum, length - sum)) > 0)
                            sum += count;  // sum is a buffer offset for next reading
                    }
                    finally
                    {
                        fileStream.Close();
                    }
                    ushort[] tmp = new ushort[(sum+1)/2];
                    int shortLength = 0;
                    for (int i = 0; i < sum; i+=2)
                    {
                        tmp[shortLength] = BitConverter.ToUInt16(buffer, i);
                        shortLength++;
                    }

                    short nErrorNo = m_camera.store_data_2_camera_memory(tmp, (uint)((sum+1)/2));
                    if (nErrorNo == CS_API.CS_OK)
                    {
                        Console.WriteLine("Successfully sent data to camera memory");
                    }
                    else
                    {
                        API_Test.textattr(API_Test.Color.RED);
                        Console.WriteLine("Sending of data to camera memory failed!\n");
                        StringBuilder errorText = new StringBuilder("", CS_API.MAX_ERROR_TEXT_LENGTH);
                        short res = m_camera.get_error_text(nErrorNo, errorText);
                        Console.WriteLine("Error:" + errorText);
                    }
                }
                else
                {
                    API_Test.textattr(API_Test.Color.RED);
                    Console.Write("\n\nNo valid filename was given!");
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }

        public static void getCameraDataFromCameraMemory()
        {
            if (m_isConnected)
            {
                Console.Write("Please enter the file name to store data to(including path):\n ");
                string fileName = Console.ReadLine();
                if (fileName.Length > 0)
                {
                    FileStream fileStream = new FileStream(fileName, FileMode.OpenOrCreate, FileAccess.Write);
                    try{
                        //       string traceMsg = "";
                        uint MAX_LENGTH = 32762;
                        ushort userDataLength = 0;
                        ushort[] userData = new ushort[MAX_LENGTH];

                        BinaryWriter writer = new BinaryWriter(fileStream);
                        ////////////////////////////////////////////////////////////////////////////////////////////
                        short nReturn = 0;

                        userDataLength = m_camera.get_data_from_camera_memory(userData, MAX_LENGTH);
                        if (nReturn >= 0)
                        {
                            for (uint i = 0; i < userDataLength; i++)
                            {
                                writer.Write(userData[i]);
                            }
                            writer.Close();
                            fileStream.Close();
                            Console.WriteLine("\n\nSuccessfully got user data from camera");
                        }
                        else
                        {
                            Console.WriteLine("\n\nError getting user data from camera");
                        }
                    }
                    finally
                    {
                        fileStream.Close();
                    }
                }
            }
            else
                Console.Write("\n !!! Camera is not connected !!!\n");
        }


        //////////////////////////////////////////////////////////////////////////
        /// Get the connected cameras and list the information
        //////////////////////////////////////////////////////////////////////////
        public static void autoDetectCameras()
        {
            if (!m_isConnected)
            {
                Console.WriteLine("\nDetecting cameras.....");
                short nReturn = 0;
                // Define where to search for cameras
                bool searchOnCamLink = true;
                bool searchOnRS232 = true;
                nReturn = m_camera.autodetect_cameras(searchOnCamLink, searchOnRS232);
                if (nReturn > 0)
                {
                    Console.WriteLine("\nDetected {0} cameras!!", nReturn);
                    CS_CAMERA_INFORMATION_STRUCT camInfo = new CS_CAMERA_INFORMATION_STRUCT();
                    for(uint i = 0; i < nReturn; i++)
                    {
                        m_camera.get_camera_information(i, camInfo);
                        Console.WriteLine("Info({0}): SerialNumber: {1}", i, camInfo.cameraSerialNumber);
                        if(camInfo.camLinkPortNo != -1)
                            Console.WriteLine("Connected to CamLink PortNo({0})", camInfo.camLinkPortNo);
                        else if (camInfo.serialPortNo != -1)
                            Console.WriteLine("Connected to Serial PortNo({0})", camInfo.serialPortNo);
                        if (camInfo.productType != "\0")
                            Console.WriteLine("Product type: {0}", camInfo.productType);
                        Console.WriteLine("\n-------------------------------------------------\n");
                        // More information can be retrieved! Just take a look at the CS_CAMERA_INFORMATION_STRUCT
                    }
                }
                else
                {
                    Console.WriteLine("No cameras detected !!!");
                }

            }
            else
                Console.Write("\n !!! Camera should not be connected when using autodetect!!!\n");
        }

        //////////////////////////////////////////////////////////////////////////
        /// List all available interfaces in the system
        //////////////////////////////////////////////////////////////////////////
        public static void list_availableInterfaces()
        {
            Int16 interfaceCount = m_camera.get_number_of_available_interfaces();
            Console.WriteLine("Found following interfaces(CL:{0}, RS232:{1})", m_camera.get_no_of_cameraLink_ports(), m_camera.get_no_of_rs232_ports());
            for (Int16 i = 0; i < interfaceCount; i++)
            {
                StringBuilder description = new StringBuilder("", CS_API.MAX_ERROR_TEXT_LENGTH);
                short res = m_camera.get_interface_description(i, description);
                Console.WriteLine("Interface({0}):{1}",i, description);
            }
        }
        
    }
}
