//////////////////////////////////////////////////////////////////////////////////////////////
/// \class CChromasensCamera
/// Description:	Provides an interface to communicate with Chromasens cameras.<BR>
///             	Different interfaces (RS232, CameraLink, Ethernet, FireWire) can be chosen.<BR>
///             	Communication can be done via convenience functions or by assembling HSI- commands. <BR>
///
/// Author:			Sczech / CHROMASENS GmbH Konstanz <BR>
///
/// Date:			June, 20, 2012
//	Version:		1.0
//////////////////////////////////////////////////////////////////////////////////////////////

#ifndef CHROMASENS_CAMERA_IF_H
#define CHROMASENS_CAMERA_IF_H

	// Import/Export definitions for the DLL-generation
	#ifdef CSCAM_EXPORT
	 #define CAM_IF __declspec(dllexport)
	#else 
	 #define CAM_IF __declspec(dllimport)
	#endif
 
	////////////////////////////////////////////////////////////////////////
	/// Includes, namespaces, etc.
	#include <string>
	using namespace std;

	namespace CSAPI
	{
	/////////////////////////////////////////////////////////////////////////////////
	/// Declarations of used enums
	/////////////////////////////////////////////////////////////////////////////////
	enum CS_WHITE_BALANCING_MODE {
		GAIN_CONTROL_USING_AREA_RANGE=1,
		USE_MAXIMUM_VIDEO_LEVEL = 2, // Not supported yet
		ODD_EVEN_CORRECTION_BY_LINE_AVG = 4// Not supported yet
	};

	enum CS_AVERAGING {
		NO_AVERAGING = 0,
		USE_2_SAMPLES_FOR_AVERAGING,
		USE_4_SAMPLES_FOR_AVERAGING,
		USE_8_SAMPLES_FOR_AVERAGING,
		USE_16_SAMPLES_FOR_AVERAGING,
		USE_32_SAMPLES_FOR_AVERAGING // 32 Samples averaging not valid for encoder averaging! 
	};

	//used for the setting of the multi-tab mode in the reference struct
	enum CS_MULTI_TAP_MODE{
		FRONT_TAP_IS_MASTER = 0,
		REAR_TAP_IS_MASTER,
		INDEPENDENT_WHITE_BALANCING,
		AUTOMATIC_MASTER_DETECTION
	};

	// used for the setting of the analog coarse gain
	enum CS_COARSE_GAIN{
		MINUS_3DB = 0,
			_0_DB,
			_3_DB,
			_6_DB
	};

	// used to set the CDS gain of the wave camera
	enum CS_CDS_GAIN{
		CDS_X1 = 0,
		CDS_X2 
	};

	// used to set the sensitivity of the sensor
	enum CS_SENSITIVITY{
		HIGH_DYNAMIC = 0,
		HIGH_SENSITIVITY 
	};

	// used to set the synchronization of the encoder mode
	enum CS_ENCODER_SYNC_MODE{
		STANDARD_ENCODER = 1,
		LINE_TRIGGER = 5
	};

	// used to set the behavior of the encoder signal evaluation
	enum CS_ENCODER_CHANNELS{
		ONE_CHANNEL_FULL_STEP = 0,
		ONE_CHANNEL_2EDGES_PER_STEP,
		TWO_CHANNEL_4EDGES_PER_STEP
	};

	// used to set which speed mode for the cameraLink connection is used
	enum CS_CAMLINK_SPEED {
		CL_STANDARD=0,	// Decreases the pixel clock of the camera Link transmission: 72,86MHz
		CL_HIGH_SPEED,	// Regular Pixel clock used. Allpixa: 85MHz
		CL_REDUCED_SPEED// Reduced pixel clock (63,75MHz) in order to avoid transmission problems on long cable lengths
	};

	// used to set which mode of the cameraLink connection should be used
	enum CS_CAMLINK_MODE{
		CL_BASE_MODE=0,				// Speed of the camera is limited to a cameraLink base connection
		CL_MEDIUM_MODE=1,			// 2XE CameraLink format
		CL_MEDIUM_MODE_1X2=2,		// 1X2 Raw format (2 tabs 24 bits) only supported by allPIXA pro
									// allPIXA pro Mono: 1X4 format (4 tabs each with 8 bits information) 
		CL_FULL64_1X8_RAW=3,		// 1x8 raw (8 tabs each with 8 bits information), RGB with 8 color planes per cameraLink clock (only allPIXA pro))
									// allPIXA pro Mono: 1X8 
		CL_FULL80_1X8=4,			// 1x8 raw, (8 tabs each with 10 bits information) RGB with 8 color planes per cameraLink clock (only allPIXA pro)
		CL_FULL80_1X10=5,			// 1x10 raw, (10 tabs each with 8 bits information) RGB with 10 color planes per cameraLink clock (only allPIXA pro)
		CL_BASE_1X3=6,				// 1x3 raw, (3 tabs with 8bits information) Only used for the allPIXA pro and gray modes! This mode is not available for color transmission

		// Definitions for the wave camera
		CL_BASE_MODE_WAVE = 0,					// 1x RGB (1 tabs with 8 bits information) RGB (only allPIXA wave))
		CL_MEDIUM_1X4_MONO_WAVE = 1,			// 1x4 mono (4 tabs with 8 bits information) gray (only allPIXA wave))
		CL_MEDIUM_1X4_4T_12BIT_MONO_WAVE = 2,	// 1x4 mono (4 tabs with 12 bits information) gray (only allPIXA wave))
		CL_MEDIUM_1X2_2T_8BIT_RGB_WAVE = 3,		// 1x2 RGB (2 tabs with 12 bits information), RGB (only allPIXA wave))
		CL_MEDIUM_1X4_1T_12BIT_RGB_WAVE = 4,	// 1x4 RGB (4 tabs with 12 bits information), RGB (only allPIXA wave))
		CL_FULL64_1X8_8T_8BIT_RAW_WAVE =5,		// 1x8 raw (8 tabs each with 8 bits information), RGB or gray (only allPIXA wave))
		CL_FULL80_1X10_10T_8BIT_RAW_WAVE = 6,	// 1x10 raw, (10 tabs each with 8 bits information) RGB or gray (only allPIXA wave)
		CL_FULL80_1X8_8T_10BIT_RAW_WAVE = 7		// 1x8 raw, (8 tabs each with 10 bits information) RGB or gray (only allPIXA wave)
	};

	// Used to set the grey video output mode
	enum CS_VIDEO_OUTPUT_MODE {
		_3x8_BIT_COLOR_PARALLEL = 0,
		_2x8_BIT_GREY_FIRST_CL_PORT,
		_2x10_BIT_GREY_FIRST_CL_PORT,
		_2x12_BIT_GREY_FIRST_CL_PORT,
		_2x8_BIT_GREY_CL_DUAL_BASE =6,
		_2x10_BIT_GREY_CL_DUAL_BASE,
		_2x12_BIT_GREY_CL_DUAL_BASE,

		// Definitions for the wave camera
		COLOR = 0,
		_3X8_BIT_MONO_BASE,
		_4X8_BIT_MONO_MEDIUM,
		_4X12_BIT_MONO_MEDIUM
	};

	// Indicates how the data of a camera setting should be saved to disk
	enum CS_SETTINGS_FORMAT {
		FORMAT_BINARY = 0, // Binary data format will be a HSI command set
		FORMAT_XML		   // XML data format which can be read and displayed by common editors
	};

	enum CS_TEST_PATTERN {
		TEST_PATTERN_OFF = 0,
		GREY_RAMP_IN_CCD_DIR,
		GREY_RAMP_IN_TRANSPORT_DIR,
		INPUT_RAMP_ON_GREEN_CHANNEL,
		SEQUENCE_OF_PATTERNS,
		CHANGE_VIDEO_LEVEL_AT_EVERY_PIXEL
	};

	enum CS_REFERENCE_MODE {
		BLACK_LEVEL_CORRECTION = 0,
		SHADING_CORRECTION
	};

	enum CS_CALC_REFERENCE_MODE{
		AUTOMATIC_DETECTION=0,
		MANUAL_SETTING
	};

	enum CS_REFERENCE_PLANE_DEFINITION{
		RED_REF_PLANE = 0,
		GREEN_REF_PLANE,
		BLUE_REF_PLANE,
		MONOCHROME_REF_PLANE,
		SPECIAL_REF_PLANE1,
		SPECIAL_REF_PLANE2,
		SPECIAL_REF_PLANE3,
		SPECIAL_REF_PLANE4,
		SPECIAL_REF_PLANE5,
		SPECIAL_REF_PLANE6,
		SPECIAL_REF_PLANE7,
		SPECIAL_REF_PLANE8,
		SPECIAL_REF_PLANE9
	};

	enum CS_REFERENCE_VERSION{
		CS_GENERAL_CAM_REF_VERSION = 0,
		CS_ALLPIXA_WAVE_VERSION =	 10
	};

	enum CS_TRIGGER_MODE {
		FREE_RUNNING = 0,
		START_CONDITION_ONLY,
		START_STOP_CONDITION = 3
	};

	enum CS_TRIGGER_INPUT {
		LIGHT_BARRIER_0 = 0,
		LIGHT_BARRIER_1,
		LIGHT_BARRIER_2,
		LIGHT_BARRIER_3
	};

	enum CS_TRIGGER_CONDITION {
		RAISING_EDGE = 0,
		FALLING_EDGE,
		CONDITION_NOT_DEFINED
	};

	enum CS_MASTER_SLAVE_OPERATION {
		USE_SELECTION_FROM_SETTING = 0,  // Only used for the global master/slave setting
		NO_MASTER_SLAVE_OPERATION = 0,	 // Only used in the local master/slave-settings
		CAMERA_IS_MASTER,
		CAMERA_IS_SLAVE,
		SELECT_MASTER_SLAVE_BY_INPUT
	};

	enum CS_GAIN_CONTROL_DISABLED{
		GAIN_CONTROL_RUNNING = 0,
		DISABLED_LIGHT_SWITCHED_OFF_INTERNALLY,
		DISABLED_LIGHT_SWITCHED_OFF_EXTERNALLY,
		DISABLED_BY_GAIN_CONTROL_CONDITION,
		NO_REFERENCES_SET
	};

	enum CS_OPERATING_STATE{
		DEVICE_IN_ERROR_STATE,
		DEVICE_IS_READY,
		DEVICE_IS_WARMING_UP
	};

	enum CS_PIXEL_FORMAT{
		BGR = 0,
		RGB
	};

	enum CS_LOGGINGMASK {
		LOG_ALWAYS = 0,       // some messages like errors are logged always
		LOG_APPCALLS = 1,
		LOG_PROCDATA_MIN = 2, 
		LOG_PROCDATA_DET = 4, 
		LOG_PROCDATA_SPEC = 8, 
		LOG_INTERFACEDATA = 256
	};

	enum CS_LED_FLASH_SYNC {
		NO_LED_DRIVER_SYNC = 0,
		LED_885KHZ_SYNC
	};

	enum CS_LED_FLASH {
		FLASH_DISABLED = 0,
		FLASH_ENABLED
	};

	enum CS_CAMERA_TYPE {
		ALLPIXA,
		ALLPIXA_PRO,
		ALLPIXA_WAVE
	};

	/// Length of the internal light barrier ROI
	enum CS_INTERNAL_LB_ROI_LENGTH {
		INTERNAL_LIGHT_BARRIER_ROI_LENGTH_32P = 0,
		INTERNAL_LIGHT_BARRIER_ROI_LENGTH_64P,
		INTERNAL_LIGHT_BARRIER_ROI_LENGTH_128P,
		INTERNAL_LIGHT_BARRIER_ROI_LENGTH_256P,
	};

	/// Defines which colors are relevant for the edge detection of the internal light barrier trigger function
	enum CS_INTERNAL_LB_COLOR_SELECT {
		INTERNAL_LIGHT_BARRIER_ALL_COLORS = 0,
		INTERNAL_LIGHT_BARRIER_RED_COLOR,
		INTERNAL_LIGHT_BARRIER_GREEN_COLOR,
		INTERNAL_LIGHT_BARRIER_BLUE_COLOR,
	};

	/// Defines what kind of data is going to be transmitted when requesting image data from the camera
	enum CS_IMAGE_DATA_TYPE {
		RAW_PIXEL_VALUES,		// Only the pixel values will be transmitted in the format BGR
		BMP_IMAGE				// The received data can be stored as is into a file to be displayed in an image viewer such as IrfanView
	};

	/// Definitions of XLC light controller IDs -> Only valid for allPIXA wave. Not implemented for allPIXA classic and allPIXA pro
	enum CS_XLC4_IDS {
		XLC_BROADCAST = 0, // Set this ID if you want to transmit data to all connected XLC-controllers
		XLC_ID_2 = 1,
		XLC_ID_3,
		XLC_ID_4,
		XLC_ID_5,
		XLC_ID_6,
		XLC_ID_7,
		XLC_ID_8,
		XLC_ID_9,
		XLC_ID_10,
		XLC_ID_11,
		XLC_ID_12,
		XLC_ID_13,
		XLC_ID_14,
		XLC_ID_15
	};

	/// Definitions of XLC channels -> Only valid for allPIXA wave. Not implemented for allPIXA classic and allPIXA pro
	enum CS_XLC4_CHANNELS {
		XLC_CHANNEL_A = 0,
		XLC_CHANNEL_B,
		XLC_CHANNEL_C,
		XLC_CHANNEL_D
	};

	enum CS_SCAN_DIRECTION {
		SCAN_FORWARD = 0,
		SCAN_BACKWARD
	};

	/////////////////////////////////////////////////////////////////////////////////
	/// Declarations of used constants
	/////////////////////////////////////////////////////////////////////////////////

	const INT16 MAX_CHANNELS = 12;
	const INT16 MAX_ANALOG_COARSE_GAIN = 6;
	const INT16 COLOR_CHANNELS = 3;
	const INT16 MAX_CAM_CONNECTIONS = 20;
	const INT16 MAX_SENSOR_LINES = 4; // Used for parameters belonging to the allPIXA wave camera which usually consist out of a red, green, blue and white(grey) sensor line
	const INT16 MAX_ROIS = 4;
	const INT16 PARAMETER_NOT_USED = -9999; // Indicates that this parameter is not used!

	const double SPEED_2_HIGH = 9999.0;	// Indicates that the detected speed of the encoder is too high -> Used in the operational values of the camera
	const double SPEED_2_SLOW = -9999.0; // Indicates that the detected speed of the encoder is too slow
	const double SPEED_NO_DATA = -9999.9; // No speed data received

	// Constants used to access the single fields in the channel struct
	const INT16 RED_ODD = 0;
	const INT16 RED_EVEN = 1;
	const INT16 GREEN_ODD = 2;
	const INT16 GREEN_EVEN = 3;
	const INT16 BLUE_ODD = 4;
	const INT16 BLUE_EVEN = 5;
	const INT16 RED_ODD_REAR = 6;
	const INT16 RED_EVEN_REAR = 7;
	const INT16 GREEN_ODD_REAR = 8;
	const INT16 GREEN_EVEN_REAR = 9;
	const INT16 BLUE_ODD_REAR = 10;
	const INT16 BLUE_EVEN_REAR = 11;

	// Constants used to access the single fields in the analog coarse gain struct
	const INT16 RED = 0;
	const INT16 GREEN = 1;
	const INT16 BLUE = 2;
	const INT16 WHITE = 3; // white channel only available for allPIXA wave cameras!
	const INT16 RED_REAR = 3;
	const INT16 GREEN_REAR = 4;
	const INT16 BLUE_REAR = 5;

	// Possible Connection Speeds
	const INT32 BAUD_RATE_19K = 19200;
	const INT32 BAUD_RATE_38K = 38400;
	const INT32 BAUD_RATE_56K = 57600;
	const INT32 BAUD_RATE_115K = 115200;
	const INT32 USE_MAX_BAUD_RATE = 0; // The interface will determine the fastest possible connection by itself! 
									   // -> May consume mor time to connect

	const INT32 USE_DEFAULT_PORT = 0;

	const char*	const DEFAULT_IP_ADDRESS= "192.168.128.200";

	// PortNumber for an ethernet connection
	const INT16 ETHERNET_CONNECTION = 1003;

	// Maximum and minimum setting numbers
	const UINT16 MAX_SETTING_NO =	18;
	const UINT16 MIN_SETTING_NO =	1;

	// maximum setting comment length
	const UINT16 MAX_SETTING_COMMENT_LENGTH = 128;

	// Maximum port description length for the camera connections
	const UINT16 MAX_PORT_DESCRIPTION_LENGTH = 256;

	// Maximum error description length
	const int MAX_ERROR_TEXT_LENGTH = 1000;

	// Maximum trace length
	const int MAX_TRACE_MESSAGE_TEXT_LENGTH = 15000;

//Maximum number of chars used for IP addresses
const int MAX_IP_CHARS = 16;

	// Error Codes

	// General Error Codes
	const INT16 CS_OK = 0;
	const INT16 CS_GENERAL_ERROR = -1;
	const INT16 CS_POINTER_GIVEN_IS_ZERO = -2;
	const INT16 CS_CAM_NOT_CONNECTED = -3;
	const INT16 CS_HSI_INTERPRETER_NOT_PRESENT = -4;
	const INT16 CS_FILE_OPEN_ERROR = -5;

	const INT16 CS_BAUD_RATE_NOT_SUPPORTED = -1000;
	const INT16 CS_SERIAL_CONNECTION_OPEN_ERROR = -1001;
	const INT16 CS_CAMERA_INTERFACE_MISSING = -1002;
	const INT16 CS_DATA_RECEPTION_FROM_CAM_FAILED = -1003;
	const INT16 CS_CL_PORT_NOT_PRESENT = -1004;

	// Error codes related to setting and getting camera parameters
	const INT16 CS_ERROR_PARAMETER_OUT_OF_RANGE = -2000;
	const INT16 CS_TAG_NOT_SUPPORTED = -2004;		// If this error is returned the connected camera does not support this function!
	const INT16 CS_ERROR_COMMAND_UNKNOW = -2005;
	const INT16 CS_NO_STORED_TAGS_PRESENT = -2006;
	const INT16 CS_ERROR_RETRIEVING_VALUE = -2007;
	const INT16 CS_ERROR_HSI_SEND = -2008;
	const INT16 CS_ERROR_INVALID_DATA_LENGTH = -2009;
	const INT16 CS_ERROR_INVALID_COMMAND_TYPE = -2010;

	const INT16 CS_ERROR_SETTING_NO_OUT_OF_RANGE = -2011;
	const INT16 CS_ERROR_IN_CAMERA_RESPONSE = -2012;
	const INT16 CS_ERROR_SETTING_NOT_AVAILABLE = -2013;
	const INT16 CS_ERROR_TAP_BALANCING_FAILED = -2014;
	const INT16 CS_ERROR_WHITE_BALANCING_FAILED = -2015;

	const INT16 CS_ERROR_CREATING_REFERENCE_PARAMS = -2200;
	const INT16 CS_ERROR_SENDING_REFERENCE_PARAMS = -2201;
	const INT16 CS_ERROR_SAVING_REFERENCE_PARAMS = -2202;
	const INT16 CS_ERROR_REFERENCE_NO_OUT_OF_RANGE = -2203;
	const INT16 CS_ERROR_REFERENCE_MODE_INVALID = -2204;

	const INT16 CS_ERROR_SAVING_SETTING_2_DISK = -2210;
	const INT16 CS_ERROR_TRIGGER_INPUT_NO_OUT_OF_RANGE = -2211;

	const INT16 CS_ERROR_XLC_NOT_FOUND = -2220;

	const INT16 CS_NOT_IMPLEMENTED_YET = -9999;

	char* const ERR_STRING_CAM_NOT_CONNECTED = "Camera is not connected!";

		// Bit definitions for the image insert mode
	const UINT16	INSERT_FIRST_LINE_INFO_BLOCK = 0x01;
	const UINT16	INSERT_TEST_RAMP_IN_LAST_LINE = 0x02;
	const UINT16	INSERT_CHECKSUM_IN_LAST_LINE = 0x04;
	const UINT16	INSERT_INFO_EACH_LINE = 0x08;
	const UINT16	INSERT_ACTIVATE_PIX_9_16 = 0x10;
	const UINT16	INSERT_CONTRAST_VALUE_SUM = 0x20;

	// Bit definitions for the camera trace
	const UINT16	TRACE_GENERAL_DEBUG_INFO = 0x0001;
	const UINT16	TRACE_TRANSPORT_LAYER = 0x0002;
	const UINT16	TRACE_TRANSPORT_LAYER_DETAILS = 0x0004;
	const UINT16	TRACE_WHITE_CONTROL = 0x0010;
	const UINT16	TRACE_ENVIRONMENT_VALUES = 0x80;
	const UINT16	TRACE_IMAGE_COUNTER = 0x0100;

	// constants used to set the parameter for the IO configuration struct
	const UINT16 ENCODER_DISABLE = 0x40;	// encoder function is enabled generally 
	const UINT16 ENCODER_ENABLE = 0x80;	// encoder function is enabled generally 
	const UINT16 ENCODER_ENABLE_WITH_CC2 = 0x100;	// encoder function is enabled with CC2 input 
	const UINT16 ENCODER_ENABLE_WITH_CC4 = 0x200; 	// encoder function is enabled with CC4 input 
	const UINT16 ENCODER_INVERT_SCAN_DIR = 0x800; 	// encoder inputs are inverted

	const UINT16 ENCODER_INCR0_CC1 = 0x1;			// CC1 signal is used as encoder clock 0 
	const UINT16 ENCODER_INCR0_GPIO_P0_N0 = 0x4; 	// GPIO_P0/N0 (X5-1/9) is used as encoder clock 0
	const UINT16 ENCODER_INCR0_GPIO_P2= 0x10; 		// GPIO_P2 (X5-4) signal is used as encoder clock 0
	const UINT16 ENCODER_INCR0_LOW = 0x1000;		// encoder clock 0 is set to static low
	const UINT16 ENCODER_INCR0_HIGH = 0x4000; 		// encoder clock 0 is set to static high

	const UINT16 ENCODER_INCR1_CC2 = 0x2; 			// CC2 signal is used as encoder clock 1
	const UINT16 ENCODER_INCR1_GPIO_P1_N1 = 0x8; 	// GPIO_P1/N1 (X5-2/10) is used as encoder clock 1
	const UINT16 ENCODER_INCR1_GPIO_N2= 0x20; 		// GPIO_N2 (X5-12) signal is used as encoder clock 1
	const UINT16 ENCODER_INCR1_LOW = 0x2000; 		// encoder clock 1 is set to static low
	const UINT16 ENCODER_INCR1_HIGH = 0x8000; 		// encoder clock 1 is set to static high

	const UINT16 LIGHTBARRIER_CC3 = 0x1;		// CC3 is used for LB0 + LB2
	const UINT16 LIGHTBARRIER_CC4 = 0x2; 		// CC4 is used for LB1 + LB2
	const UINT16 LIGHTBARRIER_GPIO_P0_N0 = 0x4;	// GPIO_P0/N0 (X5-1/9) is used for LB2
	const UINT16 LIGHTBARRIER_GPIO_P1_N1 = 0x8;	// GPIO_P1/N1 (X5-2/10) is used for LB2 + LB3
	const UINT16 LIGHTBARRIER_INTERNAL = 0x10; 	// Frame trigger will be generated by the camera due to the set image parameters
	const UINT16 LIGHTBARRIER_GPIO_P4 = 0x80;	// GPIO_P4 (X5-3) is used for LB2 + LB3 // Only for allPIXA pro

	const UINT16 SELECT_SCAN_DIR_CC2 = 0x100;		// Scan Dir selection via CC2
	const UINT16 SELECT_SCAN_DIR_GPIO_P1_N1 = 0x200;// Scan Dir selection via GPIO P1/N1(X5-2/10)
	const UINT16 SELECT_SCAN_DIR_GPIO_P2 = 0x400;	// Scan Dir selection via GPIO P2(X5-4)
	
	const UINT16 SELECT_MASTER_GPIO_4 = 0x01;	// GPIO_P4 (X5-3)is used to select the master camera
	const UINT16 SELECT_MASTER_CC4 = 0x02;		// CC4 signal is used to select the master camera

	const UINT16 MASTER_SLAVE_IF_X5_4_12 = 0x800;			// Master/Slave signals are transmitted via X5 over Pins 4 and 12 !Only firmware packages <P2.22 or Classic >P1.42
	const UINT16 MASTER_SLAVE_IF_X5_6_8 = 0x1000;			// Master/Slave signals are transmitted via X5 over Pins 6 and 8
	const UINT16 MASTER_SLAVE_IF_INTERNAL_X9_1_8 = 0x2000;	// Master/Slave signals are internally transmitted via X9 over Pins 1 and 8  ! Only PIXA pro
	const UINT16 MASTER_SLAVE_IF_INTERNAL_X12_2_4 = 0x4000;	// Master/Slave signals are internally transmitted via X12 over Pins 2 and 4  ! Only PIXA pro
	const UINT16 MASTER_SLAVE_IF_HUB_X5 = 0x8000;			// Master/Slave signals are transmitted via X5 over Pins 6-4 and 8-12  possibility to daisy chain cameras (More than 2)! Only PIXA pro

	const UINT16 GENERALFUNCTION_VSY = 0x1;	// GPIO_P2 (X5-4) is output and shows frame signal
	const UINT16 GENERALFUNCTION_HSY = 0x4;	// GPIO_N2 (X5-12) is output and shows line signal
	const UINT16 GENERALFUNCTION_FVAL_WHILE_FLASHING = 0x4000;	// Framing signal transmitted during Flash sequence   ! Only PIXA pro

	// Tag-formats
	const UINT16 TAG_BIN0 =	0x0000;
	const UINT16 TAG_BIN1 =	0x2000;
	const UINT16 TAG_SHORT=	0x4000;
	const UINT16 TAG_LONG =	0x6000;
	const UINT16 TAG_VAR =	0x8000;
	const UINT16 TAG_CONT =	0xA000;

	// standard command type for HSI-commands
	const char* const STD_HSICMD_TYPE = "MK";


	////////////////////////////////////////////////////////////////////////
	/// This structure will be filled when calling an open_control function with this parameter
	/// The structure contains all necessary information about the chosen interface.
	////////////////////////////////////////////////////////////////////////

	struct CS_CONNECTION_STRUCT{
		static const int MAX_CONNECTION_CHARS=200;
		INT16 portNumber;
		INT32 baudRate;
		bool doReset;
		char connectionDescription[MAX_CONNECTION_CHARS];
		char ethAddress[MAX_IP_CHARS];
		INT32 ethPort;

		CS_CONNECTION_STRUCT()
		{
			portNumber = 0;
			baudRate = BAUD_RATE_115K;
			doReset = false;
			strcpy_s(connectionDescription, MAX_CONNECTION_CHARS, "No descriptions");
		strcpy_s(ethAddress, MAX_IP_CHARS, DEFAULT_IP_ADDRESS);
			ethPort = 0;
		}
	} ;

	////////////////////////////////////////////////////////////////////////
	/// Structure used to get and set values which relate to the setting of the 
	/// different channels between camera and sensor
	/// When getting the values from the camera the "no_of_valid_entries" field 
	/// will be set to indicate if this is a camera with two tabs (12 valid entries) or 
	/// with a single tab (6 valid entries)
	/// e.g. gain, target reference values, etc.
	////////////////////////////////////////////////////////////////////////
	struct CS_CHANNEL_STRUCT{
		UINT16 no_of_valid_entries;
		UINT16 value[MAX_CHANNELS];
		CS_CHANNEL_STRUCT() 
		{
			no_of_valid_entries = 0;
			for(int i = 0; i < MAX_CHANNELS; i++)
				value[i] = 0;
		}
	} ;

	////////////////////////////////////////////////////////////////////////
	/// Structure used to get and set linear gain values 
	/// When getting the values from the camera the "no_of_valid_entries" field 
	/// will be set to indicate how many sensor lines are valid 
	/// allPIXA wave standard: 4 separate lines, red, green, blue, white
	////////////////////////////////////////////////////////////////////////
	struct CS_LINEAR_GAIN_STRUCT{
		UINT16 no_of_valid_entries;
		DOUBLE value[MAX_SENSOR_LINES];
		CS_LINEAR_GAIN_STRUCT() 
		{
			no_of_valid_entries = 0;
			for(int i = 0; i < MAX_SENSOR_LINES; i++)
				value[i] = 1.0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	// struct which is used for the configuration of the analog coarse gain of the Allpixa
	// The coarse gain is not set individually for odd and even
	// Use the constants RED, GREEN,BLUE, RED_REAR, GREEN_REAR, BLUE_REAR to access the single values
	// Supported cameras:	allPIXA Classic, allPIXA pro
	//////////////////////////////////////////////////////////////////////////
	struct CS_ANALOG_COARSE_GAIN_STRUCT{
		UINT16 no_of_valid_entries;
		UINT16 value[MAX_ANALOG_COARSE_GAIN];
		CS_ANALOG_COARSE_GAIN_STRUCT() 
		{
			no_of_valid_entries = 0;
			for(int i = 0; i < MAX_ANALOG_COARSE_GAIN; i++)
				value[i] = 0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	// struct which is used for the configuration of the sensitivity of the allPIXA sensor
	// Supported cameras:	allPIXA wave
	//////////////////////////////////////////////////////////////////////////
	struct CS_SENSOR_SENSITIVITY_STRUCT{
		UINT16 no_of_valid_entries;
		UINT16 cds_gain[MAX_SENSOR_LINES];
		UINT16 sensitivity[MAX_SENSOR_LINES];
		CS_SENSOR_SENSITIVITY_STRUCT() 
		{
			no_of_valid_entries = 0;
			for(int i = 0; i < MAX_SENSOR_LINES; i++)
			{
				sensitivity[i] = HIGH_DYNAMIC;
				cds_gain[i] = CDS_X1;
			}
		}
	} ;
	//////////////////////////////////////////////////////////////////////////
	// struct which is used for the configuration of the white reference of the camera
	//////////////////////////////////////////////////////////////////////////
	struct CS_WHITE_BALANCE_STRUCT{
		bool enable_white_balancing;		// Global switch to turn continuous white balancing on or off
		RECT white_reference_area;			// Definition where the reference values will be taken from
		INT16 frames_for_average;			// choose if the reference values should be averaged over a number of frames -> CS_WHITE_BLANCING_AVERAGING
		INT16 white_balancing_mode;			// Choose which mode for the white balancing should be selected. Currently only GAIN_CONTROL_USING_AREA_RANGE is supported
		bool synchronizeWithFrame;			// Select if the camera should do the white balancing only each frame or continuously
		bool stopGainControl;				// If this is enabled, gain control will stop 
											// if (Sum of all current channel values) < gain_control_stop_factor * average of all set point white values
		INT16 multi_tap_mode;				// Defines which tap will be used for white balancing or if this is done independently -> CS_MULTI_TAP_MODE
		float gain_control_stop_factor;		// Stops white balancing if stopGainControl is enabled an the described criteria is met
		INT16 red_target_value;				// Target values for the white balancing
		INT16 green_target_value;
		INT16 blue_target_value;
		INT16 white_target_value;			// white channel only for allPIXA wave camera, ignore when connected to allPIXA or allPIXA pro
		bool showWhiteReferenceBorders;		// This will mark the area in the image with a blue line where the white balancing will take its control values

		CS_WHITE_BALANCE_STRUCT(){
			enable_white_balancing = false;
			white_reference_area.top = 0;
			white_reference_area.left = 0;
			white_reference_area.right = 10;
			white_reference_area.bottom = 10;
			frames_for_average = NO_AVERAGING; 
			white_balancing_mode = GAIN_CONTROL_USING_AREA_RANGE;
			synchronizeWithFrame = false;
			stopGainControl = false;
			multi_tap_mode = AUTOMATIC_MASTER_DETECTION;
			gain_control_stop_factor = 1.0;
			red_target_value = 900;
			green_target_value = 900;
			blue_target_value = 900;
			white_target_value = 900;
			showWhiteReferenceBorders = false;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the encoder settings
	//////////////////////////////////////////////////////////////////////////
	struct CS_ENCODER_STRUCT{
		bool enable_ecoder;
		DOUBLE um_per_increment;
		DOUBLE vertical_resolution_dpi;
		UINT16 synchronisation_mode;
		UINT16 encoder_channels; // This variable only has an effect if the synchronization mode is set to STANDARD_ENCODER
		UINT16 line_trigger_reduction;
		UINT16 averaging; // values 0, 2, 4, 8, 16 are permissible. 0 = averaging off

		CS_ENCODER_STRUCT(){
			enable_ecoder = false;
			um_per_increment = 100.0;
			vertical_resolution_dpi = 300.0;
			synchronisation_mode = LINE_TRIGGER;
			encoder_channels = ONE_CHANNEL_FULL_STEP; 
			line_trigger_reduction = 0;
			averaging = 0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the trigger settings
	//////////////////////////////////////////////////////////////////////////
	struct CS_TRIGGER_STRUCT{
		INT16 trigger_mode;
		UINT16 scan_lines_after_stop;
		bool stop_after_max_no_of_scan_lines;
		UINT16 max_no_of_scan_lines;
		UINT16 no_of_suppressed_lines;
		INT16 trigger_input;
		INT16 start_condition;
		INT16 stop_condition;
		INT16 master_slave_operation;

		// Initialization of the struct
		CS_TRIGGER_STRUCT(){
			trigger_mode = FREE_RUNNING;
			scan_lines_after_stop = 0;
			stop_after_max_no_of_scan_lines = false;
			max_no_of_scan_lines = 4000;
			no_of_suppressed_lines = 0;
			trigger_input = LIGHT_BARRIER_0;
			start_condition = RAISING_EDGE;
			stop_condition = FALLING_EDGE;
			master_slave_operation = NO_MASTER_SLAVE_OPERATION;
		}

	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the parameter for the internal light barrier 
	/// -> Camera triggers according to the content of the image!
	//////////////////////////////////////////////////////////////////////////
	struct CS_INTERNAL_LIGHT_BARRIER_STRUCT{
		INT16 roi_start;
		INT16 roi_length;
		INT16 color_selection_for_edge_detection;
		INT16 mark_roi_in_image;
		INT16 rising_edge_level;
		INT16 falling_edge_level;

		// Initialization of the struct
		CS_INTERNAL_LIGHT_BARRIER_STRUCT(){
			roi_start = 500;
			roi_length = INTERNAL_LIGHT_BARRIER_ROI_LENGTH_32P;
			color_selection_for_edge_detection = INTERNAL_LIGHT_BARRIER_ALL_COLORS;
			mark_roi_in_image = 0;
			rising_edge_level = 100;  // Value in the 8bit range
			falling_edge_level = 200; // Value in the 8bit range
		}

	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the video output parameters
	//////////////////////////////////////////////////////////////////////////
	struct CS_VIDEO_OUTPUT_STRUCT{
		INT16 video_output_mode;
		float color_weight_red;
		float color_weight_green;
		float color_weight_blue;
		bool swap_color_channels_red_blue;
		INT16 insert_mode;						//! Use the constants defined with INSERT_ to bitwise set the information you want to show in the image
		INT16 position_eachline_data;
    
		// Initialization of the struct
		CS_VIDEO_OUTPUT_STRUCT(){
			video_output_mode = _3x8_BIT_COLOR_PARALLEL;
			color_weight_red = 1.0;
			color_weight_green = 1.0;
			color_weight_blue = 1.0;
			swap_color_channels_red_blue = false;
			insert_mode = 0;
			position_eachline_data = 0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the image parameters (Do not use with allPIXA wave)
	//////////////////////////////////////////////////////////////////////////
	struct CS_IMAGE_PARAMETER_STRUCT{
		UINT32 img_height;
		UINT16 img_width;
		UINT16 first_scan_line_delay; // only used if in frame scan mode -> delays the start of the image if in frame scan mode 
		// Initialization of the struct
		CS_IMAGE_PARAMETER_STRUCT() {
			img_height = 500;
			img_width = 500;
			first_scan_line_delay = 0; 
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the ROI/image parameters (Only use with allPIXA wave)
	//////////////////////////////////////////////////////////////////////////
	struct CS_ROI_PARAMETER_STRUCT{
		bool roi_active[MAX_ROIS];
		UINT32 roi_start[MAX_ROIS];
		UINT32 roi_width[MAX_ROIS];
		UINT32 img_height;
		UINT16 first_scan_line_delay; // only used if in frame scan mode -> delays the start of the image if in frame scan mode 
		
		// Initialization of the struct
		CS_ROI_PARAMETER_STRUCT() {
			for(int i = 0; i < MAX_ROIS; i++)
			{
				roi_active[i] = false;
				roi_start[i] = 500;
				roi_width[i] = 500;
			}
			roi_active[0] = true; // First ROI enabled by default
			img_height = 500;
			first_scan_line_delay = 0; 
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the image processing parameters
	//////////////////////////////////////////////////////////////////////////
	struct CS_IMAGE_PROCESSING_PARAMETER_STRUCT{
		bool mirror_image_horizontally;
		bool use_color_conversion_matrix;
		bool use_linearisation_table;
		bool use_keystone_correction;
		DOUBLE keystone_pixel_shift; 
		UINT16 keystone_correction_width; 
		UINT16 select_color_conversion_matrix; 

		// Initialization of the struct
		CS_IMAGE_PROCESSING_PARAMETER_STRUCT() {
			mirror_image_horizontally = false;
			use_color_conversion_matrix = false;
			use_linearisation_table = false;
			use_keystone_correction = false;
			keystone_pixel_shift = 0.0; 
			keystone_correction_width = 1; 
			select_color_conversion_matrix = 0; 
		}

	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the phsical setup parameters
	//////////////////////////////////////////////////////////////////////////
	struct CS_PHYSICAL_SETUP_STRUCT {
		DOUBLE rgb_line_distance;
		UINT16 scan_direction;

		// Initialization of the struct
		CS_PHYSICAL_SETUP_STRUCT (){
			rgb_line_distance = 4.0;
			scan_direction = 0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the integration time parameters
	//////////////////////////////////////////////////////////////////////////
	struct CS_INTEGRATION_TIME_STRUCT{
		UINT32 integration_time_in_ns;
		bool  use_line_period_feature;	// if this feature is used you can set a shorter integration 
										// time than the line period 
										// If using an allPIXA wave camera this parameter is always active and can not be set!
		UINT32 line_period_in_ns;
		// Initialization of the struct
		CS_INTEGRATION_TIME_STRUCT() {
			integration_time_in_ns = 50000;
			use_line_period_feature = false; 
			line_period_in_ns = 50000;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the brightness, contrast and gamma correction of the camera
	//////////////////////////////////////////////////////////////////////////
	struct CS_BRIGHTNESS_CONTRAST_STRUCT{
		bool useBrightnessAndContrast;
		INT16 brightness[MAX_SENSOR_LINES];
		DOUBLE contrast[MAX_SENSOR_LINES];
		DOUBLE gamma_correction;
		// Initialization of the struct
		CS_BRIGHTNESS_CONTRAST_STRUCT() {
			useBrightnessAndContrast = false;
			for(int i = 0; i < MAX_SENSOR_LINES; i++)
			{
				brightness[i] = 0;
				contrast[i] = 1.0;
			}
			gamma_correction = 1.0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the reference parameters like shading(flat field) or black level compensation
	//////////////////////////////////////////////////////////////////////////
	struct CS_REFERENCE_PARAMETER_STRUCT{
	
		bool bUseBlackLevelCorrection;	// Choose if the black level correction should be active
		bool bUseShadingCorrection;		// Choose if the shading correction should be active
		UINT16 nBlackLevelReferenceNo;	// No of the black level reference to apply: set to 0 or 1
		UINT16 nShadingReferenceNo;		// No of the shading reference to apply: set to 0 or 1

		// Initialization of the struct
		CS_REFERENCE_PARAMETER_STRUCT(){
			bUseBlackLevelCorrection = true;
			bUseShadingCorrection = true;
			nBlackLevelReferenceNo = 0;
			nShadingReferenceNo = 0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the parameters for the generation of a reference file for the camera
	//////////////////////////////////////////////////////////////////////////
	struct CS_CALCULATE_REFERENCE_STRUCT{
		CS_REFERENCE_MODE referenceType;
		UINT16 referenceNo;
		CS_CALC_REFERENCE_MODE calculationMode;
		unsigned char* pImage;	// Pointer to the reference image data
		UINT32 height;	// Dimensions for the image
		UINT32 width;
		UINT16 pixelFormat ; // states how the pixels are organized (BGR, RGB)
		UINT16 pixelPitch;	 // Number of bytes to move from one pixel of the same color to the next
		INT32 linePitch;	 // Number of bytes to move vertically in the same column in the image, if negative values are used, the image is bottom up with the origin in the lower left corner
		bool bSend2Camera;	// If this is set the generated reference data is sent to the camera
		char* pFileName;	// If this pointer is not set top NULL, the reference data is saved to the given file name
		UINT16 topLine;		// Defines the top position where the calculation starts in the image 
		UINT16 numberOfLines;	// Number of lines to use for the reference generation
		UINT16 targetValue;	// Only used when calculating the shading/flat field reference. Refers to the target for which the reference will be generated(Max. brightness) 
		bool doExtrapolation;	// If the image does not contain complete valid data, extrapolate to the edges
		UINT16 leftExtrapolationStart;	// Position where to start the extrapolation on the left
		UINT16 rightExtrapolationStart;	// Position where to start the extrapolation on the right
		UINT16 numberOfColumnsForExtrapolation;	// Extrapolation is done in a linear way. This will determine the number of columns to calculate the slope for the brightness decrease/increase
												// used for the extrapolation
		//////////////////////////////////////////////////////////////////////////
		// These parameters must only be used in combination with allPIXA wave
		UINT16 planeType;				// If only a single color image is used, indicate which color it belongs to. 
										// References can be generated separately for each color plane
		CS_REFERENCE_VERSION referenceVersion;			// If this reference is meant to be for an allPIXA wave, set this parameter to CS_ALLPIXA_WAVE_VERSION, otherwise use CS_GENERAL_CAM_REF_VERSION
		CS_ROI_PARAMETER_STRUCT roiParameters;	// If any ROIs in your camera are set, transfer these settings into this struct, otherwise the reference calculated will not correct the corresponding pixels!
												// first_scan_line_delay and imgHeight are not used in this context!

		CS_CALCULATE_REFERENCE_STRUCT(){
			referenceType = BLACK_LEVEL_CORRECTION;
			referenceNo = 0;
			calculationMode = MANUAL_SETTING;
			pImage = NULL;
			height = 100;
			width = 100;
			pixelFormat = BGR;
			pixelPitch = 3;
			linePitch = 300;
			bSend2Camera = true;
			pFileName = NULL;
			topLine = 0;
			numberOfLines = 100;
			UINT16 targetValue=255;
			doExtrapolation = false;
			leftExtrapolationStart = 0;
			rightExtrapolationStart = (UINT16)99999;
			numberOfColumnsForExtrapolation = 100;
			planeType = MONOCHROME_REF_PLANE;
			referenceVersion = CS_GENERAL_CAM_REF_VERSION;
		}
	};

	//////////////////////////////////////////////////////////////////////////
	/// Structure for the test pattern settings
	//////////////////////////////////////////////////////////////////////////
	struct CS_TEST_PATTERN_STRUCT{
		UINT16 nTestPattern;
		UINT16 nTestPatternLevel;
		CS_TEST_PATTERN_STRUCT(){
			nTestPattern = TEST_PATTERN_OFF;
			nTestPatternLevel = 0;
		}
	};


	//////////////////////////////////////////////////////////////////////////
	/// Structure for the CameraLink settings
	//////////////////////////////////////////////////////////////////////////
	struct CS_CAMLINK_PROPERTIES{
		CS_CAMLINK_MODE camLinkMode;
		CS_CAMLINK_SPEED camLinkSpeed;

		CS_CAMLINK_PROPERTIES(){
			camLinkMode = CL_MEDIUM_MODE;
			camLinkSpeed = CL_HIGH_SPEED;
		}
	};

	//////////////////////////////////////////////////////////////////////////
	/// Structure for the current operating values of the camera
	//////////////////////////////////////////////////////////////////////////
	struct CS_CAMERA_OPERATING_VALUES_STRUCT{
		static const int MAX_INTERNAL_VOLTAGES = 6;

		// Inputs
		static const int LB_0			= 0x01;
		static const int LB_1			= 0x02;
		static const int LB_2			= 0x04;
		static const int LB_3			= 0x08;

		INT32 inputStatus;	 // Light barrier state
		INT32 gainCtrlDisableState;
		INT32 imageCount;
		INT32 nCurrentWhiteRef[MAX_CHANNELS];
		INT32 nCurrentGain[MAX_CHANNELS];
		long syncIntegrationTime;
		INT32 cameraState;
		INT32 errorState;
		INT32 ledState;
		double currentTransportSpeed;
		INT16 currentScanDir;

		INT32 internalVoltage[MAX_INTERNAL_VOLTAGES];

		INT32 voltageLed[5];
		INT32 inputVoltage;
		INT16 temperatureBoard;
		INT16 temperatureLEDController;
		INT16 temperatureSensor;
		INT16 temperatureLed;

		INT16 masterSlaveMode;

		// Constructor of operating values struct
		CS_CAMERA_OPERATING_VALUES_STRUCT()
		{
			inputStatus = 0;
			gainCtrlDisableState = GAIN_CONTROL_RUNNING;
			imageCount = 0;
			syncIntegrationTime = 0;
			cameraState = DEVICE_IS_READY;
			errorState = 0;
			ledState = DEVICE_IS_READY;
			currentTransportSpeed = 0.0;
			currentScanDir = SCAN_FORWARD; // USE enum CS_SCAN_DIRECTION to determine the scan direction
			inputVoltage = 0;
			temperatureBoard = 0;
			temperatureLed = 0;
			temperatureLEDController = 0;
			temperatureSensor = 0;

			masterSlaveMode = NO_MASTER_SLAVE_OPERATION; // Only the first three states are possible!
														 // This entry will show how the camera is currently operating
														 // NO_MASTER_SLAVE_OPERATION
														 // CAMERA_IS_MASTER
														 // CAMERA_IS_SLAVE
			for(INT32 i = 0; i < MAX_CHANNELS; i++)
			{
				nCurrentWhiteRef[i] = 0;
				nCurrentGain[i] = 0;
			}
			for(INT32 i = 0; i<5;i++)
			{
				voltageLed[i] = 0;
			}
			for(INT32 i = 0; i<MAX_INTERNAL_VOLTAGES;i++)
			{
				internalVoltage[i] = 0;
			}
		
		};
	};

	//////////////////////////////////////////////////////////////////////////
	/// Structure for the camera informations
	//////////////////////////////////////////////////////////////////////////
	struct CS_CAMERA_INFORMATION_STRUCT{
		static const int MAX_CAMINFO_TEXT_LENGTH = 42;
		static const int MAX_CAM_SERIAL_TEXT_LENGTH = 12;

		INT16 firmwareVersion[3];
		INT16 fpgaVersion;
		char firmwareDescription[MAX_CAMINFO_TEXT_LENGTH];
		char fpgaDescription[MAX_CAMINFO_TEXT_LENGTH];
		char cameraSerialNumber[MAX_CAM_SERIAL_TEXT_LENGTH];
		INT16 packetVersion;
		char packetDescription[MAX_CAMINFO_TEXT_LENGTH];
		bool packetConsistent;

		char fullInterfaceDescription[MAX_CAMINFO_TEXT_LENGTH];
		char productType[MAX_CAMINFO_TEXT_LENGTH];
		int productTypeId; // serial number first part
		int productSerial; // serial number second part
		int serialPortNo;
		int camLinkPortNo;
		char ethAddress[MAX_IP_CHARS];
		int ethPort;
	
		// Constructor of camera information struct
		CS_CAMERA_INFORMATION_STRUCT()
		{
			firmwareVersion[0] =	0x00;
			firmwareVersion[1] =	0x00;
			firmwareVersion[2] =	0x00;
			fpgaVersion =		0x00;
			packetVersion = 0x00;
			packetConsistent = true;
			memset(packetDescription, 0, MAX_CAMINFO_TEXT_LENGTH);
			memset(firmwareDescription, 0, MAX_CAMINFO_TEXT_LENGTH);
			memset(fpgaDescription, 0, MAX_CAMINFO_TEXT_LENGTH);
			memset(cameraSerialNumber, 0, MAX_CAM_SERIAL_TEXT_LENGTH);
			memset(fullInterfaceDescription, 0, MAX_CAMINFO_TEXT_LENGTH);
			memset(productType, 0, MAX_CAMINFO_TEXT_LENGTH);
			productSerial = -1;
			serialPortNo = -1;
			camLinkPortNo = -1;
			productTypeId = -1;
			memset(ethAddress, 0, MAX_IP_CHARS);
			ethPort=-1;
		};
	};

	//////////////////////////////////////////////////////////////////////////
	/// Structure for the LED flash control (only ALLPIXA pro)
	//////////////////////////////////////////////////////////////////////////
	struct CS_LED_FLASH_CONTROL_STRUCT{
		static const int MAX_LED_FLASH_PATTERNS = 4;
		static const int MAX_OUTPUTS = 4;

		UINT16 enableFlashControl;										// Outputs only get triggered when this flag is set to enabled
		UINT16 numberOfLinePatterns;									// Number of different flash scenarios
		double flashSequenceTime;										// time in us for the whole flash sequence
		double flashDefinitions[MAX_LED_FLASH_PATTERNS][MAX_OUTPUTS];	// Definitions of the times when an output is switched on
		UINT16 ledDriverSync;											// If enabled the camera provides a synchronisation clock of 885 kHz

		// Constructor of LED flash control struct
		CS_LED_FLASH_CONTROL_STRUCT()
		{
			enableFlashControl = FLASH_DISABLED;
			numberOfLinePatterns = 1;
			flashSequenceTime = 1.0;	
			ledDriverSync = NO_LED_DRIVER_SYNC;
			for(int i = 0; i < MAX_LED_FLASH_PATTERNS; i++)
			{
				for(int outputs = 0; outputs < MAX_OUTPUTS; outputs++)
				{
					flashDefinitions[i][outputs] = 0.0;
				}
			}
		};
	};
	//////////////////////////////////////////////////////////////////////////
	/// Structure to set the logging parameters
	//////////////////////////////////////////////////////////////////////////
	struct CS_LOGGING_PARAM_STRUCT {
		char *  pLogFileName;
		UINT32 OutputToDbgViewer;
		UINT32 LoggingMask;
		UINT32 InterfaceDataLength;
		UINT32 TimeStamp;

		CS_LOGGING_PARAM_STRUCT () {
			pLogFileName = NULL;
			OutputToDbgViewer = 0;
			LoggingMask=0;
			InterfaceDataLength=0;
			TimeStamp=0;
		}
	};

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set IO configuration parameters
	//////////////////////////////////////////////////////////////////////////
	struct CS_IO_CONFIG_STRUCT {
		UINT16 encoder_config;
		UINT16 encoder_incr0;
		UINT16 encoder_incr1;
		UINT16 select_scan_dir;
		UINT16 light_barrier;
		UINT16 select_master;
		UINT16 master_slave_interface;
		UINT16 general_functions;

		// Initialization of the struct
		CS_IO_CONFIG_STRUCT () {
			encoder_config = ENCODER_ENABLE;
			encoder_incr0 = ENCODER_INCR0_CC1;
			encoder_incr1 = ENCODER_INCR1_CC2;
			select_scan_dir = 0;
			light_barrier = LIGHTBARRIER_CC3;
			select_master = SELECT_MASTER_GPIO_4;
			master_slave_interface = MASTER_SLAVE_IF_X5_4_12;
			general_functions = 0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Structure to set XLC illumination parameters
	//////////////////////////////////////////////////////////////////////////
	struct CS_XLC_PARAMETER_STRUCT {
		WORD current;
		WORD channelSelection4Current;
		WORD channelActive;
		DWORD serialNumber; // Only valid for get function! It is not possible to set the serial number!

		// Initialization of the struct
		CS_XLC_PARAMETER_STRUCT () {
			current = 0;
			channelSelection4Current = 0;
			channelActive =0;
			serialNumber = 0;
		}
	} ;

	//////////////////////////////////////////////////////////////////////////
	/// Bitmap structure 
	//////////////////////////////////////////////////////////////////////////
	typedef struct
	{
		WORD    bfType;
		UINT32  bfSize;
		WORD    bfReserved1;
		WORD    bfReserved2;
		UINT32  PixOffs;
		UINT32  biSize;
		UINT32  biWidth;
		UINT32  biHeight;
		WORD    biPlanes;
		WORD    biBitCount;
		UINT32  biCompression;
		UINT32  biSizeImage;
		UINT32  biXPelsPerMeter;
		UINT32  biYPelsPerMeter;
		UINT32  biClrUsed;
		UINT32  biClrImportant;
	}t_BMPType;

	
	///++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	/// Declaration of the camera interface class
	///++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	#ifdef __cplusplus

	class CChromasensCameraIf{
	public:
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Constructor / Destructor
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		CChromasensCameraIf(void){};
		virtual ~CChromasensCameraIf(void){};

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions to establish a connection to a camera and retrieve information about the connection/camera
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 open_control ( CS_CONNECTION_STRUCT* pConnectInfo )=0;
		virtual INT16 open_RS232_control ( INT16 portNumber, INT32 baudRate=USE_MAX_BAUD_RATE, bool doReset=false)=0;
		virtual INT16 open_CL_control ( INT16 portNumber, INT32 baudRate=USE_MAX_BAUD_RATE)=0;
		virtual INT16 open_ETH_control ( const char* ipAddress=DEFAULT_IP_ADDRESS, INT16 portNumber=USE_DEFAULT_PORT)=0;
		virtual INT16 close_control (void)=0;
		virtual INT16 autodetect_cameras(bool searchOnCameraLink=true, bool searchOnRS232=true, bool searchOnEthernet=true)=0;
		virtual INT16 get_number_of_available_interfaces(void)=0;
		virtual INT16 get_interface_description(INT16 connectionID, char** pConnectionDescription)=0; 
		virtual INT16 get_no_of_cameraLink_ports ( void) = 0;
		virtual INT16 get_no_of_rs232_ports ( void) = 0;
		virtual INT16 determineCameraType(bool bUpdate=true) = 0;
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions to set the cameraLink Interface properties
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 set_camLink_properties(const CS_CAMLINK_PROPERTIES& camLinkProperties, bool bSend2Camera=true)=0;
		virtual INT16 get_camLink_properties(CS_CAMLINK_PROPERTIES *camLinkProperties, bool bUpdate=true)=0;

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions related to the state of the camera
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 reset_camera(void)=0;
		virtual INT16 get_error_text(INT16 nErrorNo, char** pErrorText)=0;
		virtual INT16 get_camera_state(bool bGetLastState=false, char** pErrorText=NULL)=0;
		virtual INT16 do_tap_balancing(bool bSaveWhiteCalibParams=true)=0;
		virtual INT16 perform_white_balancing(bool bSaveWhiteCalibParams=true)=0;
		virtual INT16 get_camera_operating_values(CS_CAMERA_OPERATING_VALUES_STRUCT *operatingValues)=0;
		virtual INT16 get_camera_information(CS_CAMERA_INFORMATION_STRUCT *cameraInformation)=0;
		virtual INT16 get_camera_information(unsigned int index, CS_CAMERA_INFORMATION_STRUCT *cameraInformation)=0;

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions related to calibrate/setup the camera to its environment
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 calculate_reference(const CS_CALCULATE_REFERENCE_STRUCT& calcRefParams)=0;			
		virtual INT16 create_reference_internally(const CS_REFERENCE_MODE refType, const UINT refNo, const UINT timeoutInSecconds=60)=0;
		virtual INT16 prepare_cam_4reference(const CS_REFERENCE_MODE& referencMode)=0;
		virtual INT16 set_reference_settings(const CS_REFERENCE_PARAMETER_STRUCT& referenceParameter, bool bSend2Camera=true)=0;
		virtual INT16 get_reference_settings(CS_REFERENCE_PARAMETER_STRUCT *referenceParameter, bool bUpdate=true)=0;
	
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions related to the settings of the camera
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 select_active_setting(INT16 settingNo) = 0;
		virtual INT16 get_current_setting_number(void) = 0;
		virtual INT16 burn_current_setting_to_camera(INT16 settingNo=-1)=0;
		virtual INT64 get_available_settings(void)=0;
		virtual INT16 save_current_setting_2Disk(char* filename, INT16 settingFormat=FORMAT_BINARY, UINT16 saveAsSettingNo=0)=0; 
		virtual INT16 set_setting_description(char* description, bool bSend2Camera=true)=0;
		virtual INT16 get_setting_description(char* description, bool bUpdate=true)=0;
	
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Getting and setting parameters
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

		// Treatment of Gain settings
		virtual INT16 set_gain_values(const CS_CHANNEL_STRUCT& gainValues, bool bSend2Camera=true)=0;
		virtual INT16 get_gain_values(CS_CHANNEL_STRUCT *gainValues, bool bUpdate=false)=0;

		virtual INT16 set_linear_gain_values(CS_LINEAR_GAIN_STRUCT& linearGainValues, bool bSend2Camera=true)=0; // Only valid for allPIXA wave!
		virtual INT16 get_linear_gain_values(CS_LINEAR_GAIN_STRUCT *linearGainValues, bool bUpdate=false)=0;   // Only valid for allPIXA wave!

		virtual INT16 set_coarse_gain_values(const CS_ANALOG_COARSE_GAIN_STRUCT& coarseGainValues, bool bSend2Camera=true)=0;
		virtual INT16 get_coarse_gain_values(CS_ANALOG_COARSE_GAIN_STRUCT *coarseGainValues, bool bUpdate=false)=0;

		virtual INT16 set_initial_gain_values(const CS_CHANNEL_STRUCT& gainValues, bool bSend2Camera=true)=0;
		virtual INT16 get_initial_gain_values(CS_CHANNEL_STRUCT *gainValues, bool bUpdate=false)=0;

		virtual INT16 set_sensor_sensitivity_values(const CS_SENSOR_SENSITIVITY_STRUCT& sensitivityValues, bool bSend2Camera=true)=0;
		virtual INT16 get_sensor_sensitivity_values(CS_SENSOR_SENSITIVITY_STRUCT *sensitivityValues, bool bUpdate=false)=0;

		// Configuration of settings for the reference
		virtual INT16 get_current_reference_values(CS_CHANNEL_STRUCT *referenceValues, bool bUpdate=false)=0;

		virtual INT16 set_camera_white_balancing_setup(const CS_WHITE_BALANCE_STRUCT& whiteBalanceSetup, bool bSend2Camera=true)=0;
		virtual INT16 get_camera_white_balancing_setup(CS_WHITE_BALANCE_STRUCT *whiteBalanceSetup, bool bUpdate=false)=0;

		// Encoder related settings
		virtual INT16 set_encoder_parameters(const CS_ENCODER_STRUCT& encoderSetting, bool bSend2Camera=true)=0;
		virtual INT16 get_encoder_parameters(CS_ENCODER_STRUCT *encoderSetting, bool bUpdate=false)=0;

		// Integration Time
		virtual INT16 set_integration_time(const CS_INTEGRATION_TIME_STRUCT& integrationTime, bool bSend2Camera=true)=0;
		virtual INT16 get_integration_time(CS_INTEGRATION_TIME_STRUCT *integrationTime, bool bUpdate=false)=0;

		// Set the brightness, contrast and gamma values
		virtual INT16 set_brightness_and_contrast(const CS_BRIGHTNESS_CONTRAST_STRUCT& brightnessContrast, bool bSend2Camera=true)=0;
		virtual INT16 get_brightness_and_contrast(CS_BRIGHTNESS_CONTRAST_STRUCT *brightnessContrast, bool bUpdate=false)=0;

		// Test pattern settings
		virtual INT16 set_test_pattern_settings(const CS_TEST_PATTERN_STRUCT& testPatternSetting, bool bSend2Camera=true)=0;
		virtual INT16 get_test_pattern_settings(CS_TEST_PATTERN_STRUCT *testPatternSetting, bool bUpdate=false)=0;

		// Video output parameter
		virtual INT16 set_video_output_parameters(const CS_VIDEO_OUTPUT_STRUCT& videoOutParams, bool bSend2Camera=true)=0;
		virtual INT16 get_video_output_parameters(CS_VIDEO_OUTPUT_STRUCT *videoOutParams, bool bUpdate=false)=0;

		// Image parameter 
		// Do not use the following two functions together with the allPIXA wave!
		virtual INT16 set_image_parameters(const CS_IMAGE_PARAMETER_STRUCT& imageParameters, bool bSend2Camera=true)=0;
		virtual INT16 get_image_parameters(CS_IMAGE_PARAMETER_STRUCT *imageParameters, bool bUpdate=false)=0;
		
		// Use the following two functions together with the allPIXA wave!
		virtual INT16 set_roi_parameters(const CS_ROI_PARAMETER_STRUCT& roiParameters, bool bSend2Camera=true)=0;
		virtual INT16 get_roi_parameters(CS_ROI_PARAMETER_STRUCT *roiParameters, bool bUpdate=false)=0;

		// Image processing parameters
		virtual INT16 set_image_processing_parameters(const CS_IMAGE_PROCESSING_PARAMETER_STRUCT& imageProcessingParameters, bool bSend2Camera=true)=0;
		virtual INT16 get_image_processing_parameters(CS_IMAGE_PROCESSING_PARAMETER_STRUCT *imageProcessingParameters, bool bUpdate=false)=0;

		// Trigger settings
		virtual INT16 set_camera_trigger_mode(const CS_TRIGGER_STRUCT& triggerSettings, bool bSend2Camera=true)=0;
		virtual INT16 get_camera_trigger_mode(CS_TRIGGER_STRUCT *triggerSettings, bool bUpdate=false)=0;

		// Global master slave settings, will be saved permanently and setting independent. Overrules the locally set master slave behavior
		virtual INT16 set_global_master_slave_settings(const WORD globalMasterSlaveSetting, bool bSend2Camera=true)=0;
		virtual INT16 get_global_master_slave_settings(WORD *globalMasterSlaveSetting, bool bUpdate=false)=0;

		// Setup of the internal light barrier (To switch it on or off, use the IO-configuration)
		virtual INT16 set_internal_light_barrier_settings(const CS_INTERNAL_LIGHT_BARRIER_STRUCT& internalLightBarrierSettings, bool bSend2Camera=true)=0;
		virtual INT16 get_internal_light_barrier_settings(CS_INTERNAL_LIGHT_BARRIER_STRUCT *internalLightBarrierSettings, bool bUpdate=false)=0;

		// physical setup parameters
		virtual INT16 set_camera_physical_setup (const CS_PHYSICAL_SETUP_STRUCT& physicalSetup, bool bSend2Camera=true)=0;
		virtual INT16 get_camera_physical_setup (CS_PHYSICAL_SETUP_STRUCT *physicalSetup, bool bUpdate=false)=0;

		// LED flash parameters (Only supported for allPIXA pro models)
		virtual INT16 set_led_flash_parameters (const CS_LED_FLASH_CONTROL_STRUCT& ledFlashParameters, bool bSend2Camera=true)=0;
		virtual INT16 get_led_flash_parameters (CS_LED_FLASH_CONTROL_STRUCT *ledFlashParameters, bool bUpdate=false)=0;

		// setup IO config parameters
		virtual INT16 set_IO_signal_configuration (const CS_IO_CONFIG_STRUCT& IOConfiguration, bool bSend2Camera=true)=0;
		virtual INT16 get_IO_signal_configuration (CS_IO_CONFIG_STRUCT *IOConfiguration, bool bUpdate=false)=0;

		// Store and retrieve user data to the flash memory of the camera
		virtual INT16 store_data_2_camera_memory (WORD* const  data2Store, unsigned long dataLength)=0;
		virtual UINT16 get_data_from_camera_memory (WORD* const  dataRead, unsigned long maxDataLength)=0;

		// Retrieve the current image data from the camera
		virtual INT16 get_image_line_data_from_camera(char* const imageData, const WORD maxDataLength, CS_IMAGE_DATA_TYPE dataType = RAW_PIXEL_VALUES) = 0;

		// Functions to control connected XLC4 light controllers
		// Functions only valid for the allPIXA wave camera!
		virtual INT16 get_connected_XLC_devices(WORD &connectedXLCs, bool bRescan = false)=0; // This function will return the connected XLCs as bitwise WORD
		virtual INT16 set_xlc_parameters(WORD xlcID, CS_XLC_PARAMETER_STRUCT &xlcParameters, bool bSend2Camera=true)=0;
		virtual INT16 get_xlc_parameters(WORD xlcID, CS_XLC_PARAMETER_STRUCT *xlcParameters, bool bUpdate=false )=0;

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions to send HSI-files like commands, firmware-updates, etc. to the camera
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 send_hsi_file_2_camera(const char* fileName)=0; // XML and binary files are allowed!
		virtual INT16 send_filelist_2_camera(const char* fileName, bool bShowDialog)=0; // Downloads complete lists of files to the camera! 
																					    // Need to be ASCII-files with each file listed
																						// Files need to be in the same directory as the list file
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions for internal logging and tracing of camera events, states, etc.	
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 set_trace_mode(UINT16 traceOptions, bool bSend2Camera=true)=0;
		virtual INT16 get_trace_mode(UINT16& traceOptions, bool bUpdate=false)=0;
		virtual INT16 get_trace_from_camera(char** pTraceData, INT16 &traceLength)=0;

		// Logging function for the CSAPI
		virtual INT16 set_logging_parameter (CS_LOGGING_PARAM_STRUCT loggingParams,	bool bStoreToRegistry)=0;
		virtual INT16 get_logging_parameter (CS_LOGGING_PARAM_STRUCT *loggingParams)=0;


		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions to send raw HSI-Tags to the camera
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 send_bin_tag (WORD tagID, bool value, bool sendImmediately=true) = 0;
		virtual INT16 send_short_tag (WORD tagID, WORD value, bool sendImmediately=true) = 0;
		virtual INT16 send_long_tag (WORD tagID, DWORD value, bool sendImmediately=true) = 0;
		virtual INT16 send_var_tag (WORD tagID, WORD length, WORD* pData, bool sendImmediately=true) = 0;
		virtual INT16 create_container_tag (WORD tagID, WORD length)=0;
		virtual INT16 send_stored_hsi_commands(bool bClearCommands=true)=0;

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Functions to retrieve raw HSI-tags from the camera
		/// If the update flag is set, the current values are retrieved from the camera to PC memory, otherwise the values from the last 
		/// transmission are retrieved
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 get_bin_tag (WORD tagID, bool &value, bool bUpdate=false) = 0;
		virtual INT16 get_short_tag (WORD tagID, WORD &value, bool bUpdate=false) = 0;
		virtual INT16 get_long_tag (WORD tagID, DWORD &value, bool bUpdate=false) = 0;
		virtual INT16 get_var_tag (WORD tagID, WORD length, WORD* pData, bool bUpdate=false) = 0;

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/// Low level functions to access registers on the Camera
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		virtual INT16 set_single_register( WORD offset, WORD value) = 0;
		virtual INT16 get_single_register( WORD offset, WORD &value) = 0;
	};

	#endif
	}
	#ifdef __cplusplus
	extern "C" {
	#endif
		CAM_IF CSAPI::CChromasensCameraIf* InitCsCameraIf(char* pConfigPath = NULL);
	#ifdef __cplusplus
	}
	#endif
#endif